<?php

namespace App\Http\Controllers;

use App\Models\HumanRightsCase;
use App\Services\ImportadorDerechoshumanos;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class HumanRightsAndDueDiligenceController extends Controller
{
    public function index()
    {
        $currentYear = date('Y');

        $totalForms = HumanRightsCase::where('version', $currentYear)->count();

        $FormsDe = HumanRightsCase::all();

        return view('DerechosHumanos.index', compact('FormsDe', 'totalForms'));
    }

    /**
     * Método para DataTables - retorna datos en formato JSON
     */
    public function datatables(Request $request)
    {
        try {
            $query = HumanRightsCase::query();

            // Aplicar filtros si existen
            if ($request->has('year') && $request->year != '') {
                $query->where('version', $request->year);
            }

            if ($request->has('municipio') && $request->municipio != '') {
                $query->where('municipio', 'like', '%' . $request->municipio . '%');
            }

            if ($request->has('genero') && $request->genero != '') {
                $query->where('genero', $request->genero);
            }

            // Obtener todos los registros
            $cases = $query->orderBy('created_at', 'desc')->get();

            // Formatear los datos para DataTables
            $data = $cases->map(function ($case) {
                return [
                    'id' => $case->id,
                    '_id' => $case->_id,
                    'start' => $case->start ? date('d/m/Y H:i', strtotime($case->start)) : '',
                    'end' => $case->end ? date('d/m/Y H:i', strtotime($case->end)) : '',
                    'nombre_encuestador' => $case->nombre_encuestador,
                    'dni' => $case->dni,
                    'dni_o' => $case->dni_o,
                    'socio' => $case->socio,
                    'municipio' => $case->municipio,
                    'localidad' => $case->localidad,
                    'grupo_trabajo' => $case->grupo_trabajo,
                    'telefono' => $case->telefono,
                    'genero' => $case->genero,
                    'fecha_nacimiento' => $case->fecha_nacimiento ? date('d/m/Y', strtotime($case->fecha_nacimiento)) : '',
                    'contrato_mano_obra' => $case->contrato_mano_obra,
                    'edad_jornaleros' => $case->edad_jornaleros,
                    'actividad_realizada' => $case->actividad_realizada,
                    'actividad_especifica' => $case->actividad_especifica,
                    'origen' => $case->origen,
                    'estado' => $case->estado,
                    'pais' => $case->pais,
                    'comunidad' => $case->comunidad,
                    'trabajo_forzado' => $case->trabajo_forzado,
                    'observaciones' => $case->observaciones,
                    'firma' => $case->firma,
                    '__version__' => $case->__version__,
                    'instanceID' => $case->instanceID,
                    'instanceName' => $case->instanceName,
                    '_xform_id_string' => $case->_xform_id_string,
                    '_uuid' => $case->_uuid,
                    'rootUuid' => $case->rootUuid,
                    'formhub_uuid' => $case->formhub_uuid,
                    '_status' => $case->_status,
                    '_submission_time' => $case->_submission_time ? date('d/m/Y H:i', strtotime($case->_submission_time)) : '',
                    '_submitted_by' => $case->_submitted_by,
                    'version' => $case->version,
                    'created_at' => $case->created_at ? $case->created_at->format('d/m/Y H:i') : '',
                    'updated_at' => $case->updated_at ? $case->updated_at->format('d/m/Y H:i') : '',
                ];
            });

            return response()->json($data);

        } catch (\Exception $e) {
            Log::error('Error en datatables de derechos humanos: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function importarKobo(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'year' => 'required|integer|min:2020|max:' . (date('Y') + 1),
            'auth_method' => 'required|in:basic,token',
            'auth_credential' => 'required|string',
            'second_auth_credential' => 'required_if:auth_method,basic|nullable|string',
            'asset_id' => 'required|string',
            'server_url' => 'required|url'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Datos de entrada inválidos',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $validator->validated();

            // Crear instancia del importador
            $importador = new ImportadorDerechoshumanos(
                year: $data['year'],
                assetId: $data['asset_id'],
                username: $data['auth_credential'],
                password: $data['second_auth_credential'] ?? null,
                serverUrl: $data['server_url']
            );

            // Ejecutamos la importación
            $resultado = $importador->importar();

            if ($resultado['success']) {
                Log::info('Importación exitosa desde KoBoToolbox', [
                    'year' => $data['year'],
                    'total_registros' => $resultado['total']
                ]);
            }
            
            return response()->json($resultado);

        } catch (\Exception $e) {
            Log::error('Error en importación KoBoToolbox', [
                'error' => $e->getMessage(),
                'year' => $request->year
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error interno del servidor: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener información del asset antes de importar
     */
    public function infoAsset(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'auth_method' => 'required|in:basic,token',
            'auth_credential' => 'required|string',
            'second_auth_credential' => 'required_if:auth_method,basic|nullable|string',
            'asset_id' => 'required|string',
            'server_url' => 'required|url',
            'year' => 'required|integer|min:2020'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Datos de entrada inválidos',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $validator->validated();

            $importador = new ImportadorDerechoshumanos(
                year: $data['year'],
                assetId: $data['asset_id'],
                username: $data['auth_credential'],
                password: $data['second_auth_credential'] ?? null,
                serverUrl: $data['server_url']
            );

            $info = $importador->obtenerInfoAsset();
            return response()->json($info);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener la información: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Exportar datos a Excel
     */
    public function exportar(Request $request)
    {
        try {
            $year = $request->input('year', date('Y'));

            $cases = HumanRightsCase::where('version', $year)
                ->orderBy('created_at', 'desc')
                ->get();

            // Aquí puedes implementar la exportación con Laravel Excel o similar
            // Por ahora retorno un CSV simple
            
            $filename = "derechos_humanos_{$year}_" . date('Y-m-d') . ".csv";
            
            $headers = [
                'Content-Type' => 'text/csv; charset=UTF-8',
                'Content-Disposition' => "attachment; filename=\"{$filename}\"",
            ];

            $callback = function() use ($cases) {
                $file = fopen('php://output', 'w');
                
                // BOM para UTF-8
                fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
                
                // Headers
                fputcsv($file, [
                    'ID', 'ID Kobo', 'Fecha Inicio', 'Fecha Fin', 'Encuestador', 'DNI', 
                    'Socio', 'Municipio', 'Localidad', 'Género', 'Fecha Nacimiento',
                    'Contrata Mano Obra', 'Trabajo Forzado', 'Observaciones'
                ]);

                // Data
                foreach ($cases as $case) {
                    fputcsv($file, [
                        $case->id,
                        $case->_id,
                        $case->start,
                        $case->end,
                        $case->nombre_encuestador,
                        $case->dni,
                        $case->socio,
                        $case->municipio,
                        $case->localidad,
                        $case->genero,
                        $case->fecha_nacimiento,
                        $case->contrato_mano_obra,
                        $case->trabajo_forzado,
                        $case->observaciones,
                    ]);
                }

                fclose($file);
            };

            return response()->stream($callback, 200, $headers);

        } catch (\Exception $e) {
            Log::error('Error en exportación: ' . $e->getMessage());
            return back()->with('error', 'Error al exportar: ' . $e->getMessage());
        }
    }

    /**
     * Debug de estructura - útil para desarrollo
     */
    public function debugEstructura(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'auth_method' => 'required|in:basic,token',
            'auth_credential' => 'required|string',
            'second_auth_credential' => 'required_if:auth_method,basic|nullable|string',
            'asset_id' => 'required|string',
            'server_url' => 'required|url'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Datos de entrada inválidos',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $validator->validated();

            $importador = new ImportadorDerechoshumanos(
                year: date('Y'),
                assetId: $data['asset_id'],
                username: $data['auth_credential'],
                password: $data['second_auth_credential'] ?? null,
                serverUrl: $data['server_url']
            );

            // Obtener estructura del formulario
            $estructura = $importador->debugEstructura();
            
            return response()->json([
                'success' => true,
                'estructura' => $estructura
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener estructura: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Sincronización automática
     */
    public function sincronizarAutomatico()
    {
        $config = [
            'year' => date('Y'),
            'asset_id' => config('services.kobotoolbox.DERECHOS_ASSET_UID'),
            'username' => config('services.kobotoolbox.username'),
            'password' => config('services.kobotoolbox.password'),
            'server_url' => config('services.kobotoolbox.server_url', 'https://eu.kobotoolbox.org')
        ];

        if (!$config['username'] || !$config['password'] || !$config['asset_id']) {
            Log::error('Configuración de KoboToolbox incompleta para sincronización automática');
            return response()->json([
                'success' => false,
                'message' => 'Configuración incompleta'
            ], 422);
        }

        try {
            $importador = new ImportadorDerechoshumanos(
                year: $config['year'],
                assetId: $config['asset_id'],
                username: $config['username'],
                password: $config['password'],
                serverUrl: $config['server_url']
            );

            $resultado = $importador->importar();

            Log::info('Sincronización automática completada', $resultado);
            return response()->json($resultado);

        } catch (\Exception $e) {
            Log::error("Error en sincronización automática: {$e->getMessage()}");
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }
}