<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Inspeccion;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class InspeccionController extends Controller
{
    public function index()
    {
        return view('InspeccionInterna.index');
    }

    public function getAssetInfo(Request $request)
    {
        $request->validate([
            'auth_credential' => 'required',
            'second_auth_credential' => 'required',
            'asset_id' => 'required',
            'server_url' => 'required',
        ]);

        try {
            $url = $request->server_url . '/api/v2/assets/' . $request->asset_id . '/';
            
            $response = Http::withBasicAuth(
                $request->auth_credential,
                $request->second_auth_credential
            )->get($url);

            if ($response->successful()) {
                $data = $response->json();
                
                return response()->json([
                    'success' => true,
                    'name' => $data['name'] ?? 'N/A',
                    'deployment_count' => $data['deployment__submission_count'] ?? 0,
                    'date_modified' => $data['date_modified'] ?? null,
                ]);
            }

            return response()->json([
                'success' => false,
                'message' => 'No se pudo obtener información del asset'
            ], 400);

        } catch (\Exception $e) {
            Log::error('Error al verificar asset: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function importarKobo(Request $request)
    {
        $request->validate([
            'auth_credential' => 'required',
            'second_auth_credential' => 'required',
            'asset_id' => 'required',
            'server_url' => 'required',
            'year' => 'required|integer',
        ]);

        try {
            $url = $request->server_url . '/api/v2/assets/' . $request->asset_id . '/data/?format=json';
            
            $response = Http::withBasicAuth(
                $request->auth_credential,
                $request->second_auth_credential
            )->timeout(120)->get($url);

            if (!$response->successful()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error al conectar con KoboToolbox'
                ], 400);
            }

            $koboData = $response->json()['results'] ?? [];
            
            // Filtrar por año
            $year = $request->year;
            $koboData = array_filter($koboData, function($item) use ($year) {
                $submissionTime = $item['_submission_time'] ?? '';
                return substr($submissionTime, 0, 4) == $year;
            });

            if (empty($koboData)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No se encontraron registros para el año ' . $year
                ], 404);
            }

            $imported = 0;
            $updated = 0;
            $errors = [];

            foreach ($koboData as $record) {
                try {
                    $result = $this->processAndSaveRecord($record);
                    if ($result['action'] === 'created') {
                        $imported++;
                    } else {
                        $updated++;
                    }
                } catch (\Exception $e) {
                    $errors[] = [
                        'id' => $record['_id'] ?? 'unknown',
                        'error' => $e->getMessage()
                    ];
                    Log::error('Error procesando registro: ' . $e->getMessage(), [
                        'record_id' => $record['_id'] ?? 'unknown'
                    ]);
                }
            }

            return response()->json([
                'success' => true,
                'total' => count($koboData),
                'imported' => $imported,
                'updated' => $updated,
                'errors' => count($errors),
                'error_details' => $errors
            ]);

        } catch (\Exception $e) {
            Log::error('Error en importación: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    protected function processAndSaveRecord(array $koboData)
    {
        $mappedData = $this->mapKoboFields($koboData);
        
        $inspeccion = Inspeccion::updateOrCreate(
            ['kobo_id' => $mappedData['kobo_id']],
            $mappedData
        );

        return [
            'action' => $inspeccion->wasRecentlyCreated ? 'created' : 'updated',
            'inspeccion' => $inspeccion
        ];
    }

    protected function mapKoboFields(array $data): array
    {
        return [
            // Metadatos de KoboToolbox
            'kobo_id' => $data['_id'] ?? null,
            'submission_time' => $data['_submission_time'] ?? null,
            'submitted_by' => $data['_submitted_by'] ?? null,
            'uuid' => $data['_uuid'] ?? null,
            
            // Campos del sistema
            'start' => $data['start'] ?? null,
            'end' => $data['end'] ?? null,
            'today' => $data['today'] ?? null,
            'username' => $data['username'] ?? null,
            'deviceid' => $data['deviceid'] ?? null,
            'phonenumber' => $data['phonenumber'] ?? null,
            'id_record' => $data['id_record'] ?? null,
            'hora' => $data['hora'] ?? null,
            
            // Información General
            'fecha_visita' => $data['info_general/fecha_visita'] ?? null,
            'hora_inicio' => $data['info_general/hora_inicio'] ?? null,
            'inspector' => $data['info_general/inspector'] ?? null,
            'gps_productor' => $data['info_general/gps_productor'] ?? null,
            'productor_nuevo' => $data['info_general/productor_nuevo'] ?? null,
            'otra_organizacion' => $data['info_general/otra_organizacion'] ?? null,
            'nombre_otra_organizacion' => $data['info_general/nombre_otra_organizacion'] ?? null,
            
            // Datos del Productor
            'dni' => $data['productor/dni'] ?? null,
            'id_inspeccion' => $data['productor/id_inspeccion'] ?? null,
            'socio' => $data['productor/socio'] ?? null,
            'curp' => $data['productor/curp'] ?? null,
            'rfc' => $data['productor/rfc'] ?? null,
            'ine' => $data['productor/ine'] ?? null,
            'folio_cafetalero' => $data['productor/folio_cafetalero'] ?? null,
            'folio_suri' => $data['productor/folio_suri'] ?? null,
            'municipio' => $data['productor/municipio'] ?? null,
            'localidad' => $data['productor/localidad'] ?? null,
            'grupo_trabajo' => $data['productor/grupo_trabajo'] ?? null,
            'telefono' => $data['productor/telefono'] ?? null,
            'genero' => $data['productor/genero'] ?? null,
            'fecha_nacimiento' => $data['productor/fecha_nacimiento'] ?? null,
            'fecha_ingreso' => $data['productor/fecha_ingreso'] ?? null,
            'estatus' => $data['productor/estatus'] ?? null,
            'fecha_transicion' => $data['productor/fecha_transicion'] ?? null,
            'baja_productor' => $data['productor/baja_productor'] ?? null,
            'tenencia_tierra' => $this->extractMultiple($data, 'productor/tenencia_tierra'),
            'titulo_tenencia' => $data['productor/titulo_tenencia'] ?? null,
            'papel_tenencia' => $data['productor/papel_tenencia'] ?? null,
            'participa_otra_organizacion' => $data['productor/participa_otra_organizacion'] ?? null,
            'producto_otra_organizacion' => $data['productor/producto_otra_organizacion'] ?? null,
            'nombre_otra_organizacion_001' => $data['productor/nombre_otra_organizacion_001'] ?? null,
            'familia_participa_otra_organizacion' => $data['productor/familia_participa_otra_organizacion'] ?? null,
            'familia_producto_otra_organizazacion' => $data['productor/familia_producto_otra_organizazacion'] ?? null,
            'familia_otra_organizacion' => $data['productor/familia_otra_organizacion'] ?? null,
            'familia_categoria_otra_organizacion' => $data['productor/familia_categoria_otra_organizacion'] ?? null,
            
            // Área General
            'tiene_area_autoconsumo' => $data['area_general/tiene_area_autoconsumo'] ?? null,
            'area_autoconsumo' => $data['area_general/area_autoconsumo'] ?? null,
            'autoconsumo_n_parcelas' => $data['area_general/autoconsumo_n_parcelas'] ?? null,
            'autoconsumo_quimicos' => $data['area_general/autoconsumo_quimicos'] ?? null,
            'autoconsumo_fecha_quimicos' => $data['area_general/autoconsumo_fecha_quimicos'] ?? null,
            'tiene_area_potrero' => $data['area_general/tiene_area_potrero'] ?? null,
            'area_potrero' => $data['area_general/area_potrero'] ?? null,
            'potrero_n_parcelas' => $data['area_general/potrero_n_parcelas'] ?? null,
            'tiene_area_bosque' => $data['area_general/tiene_area_bosque'] ?? null,
            'area_bosque' => $data['area_general/area_bosque'] ?? null,
            'bosque_n_parcelas' => $data['area_general/bosque_n_parcelas'] ?? null,
            'tiene_area_acahual' => $data['area_general/tiene_area_acahual'] ?? null,
            'area_acahual' => $data['area_general/area_acahual'] ?? null,
            'acahual_n_parcelas' => $data['area_general/acahual_n_parcelas'] ?? null,
            
            // Semillero y Vivero
            'sem_semillero' => $data['semillero_vivero/sem_semillero'] ?? null,
            'sem_numero_semilleros' => $data['semillero_vivero/sem_numero_semilleros'] ?? null,
            'sem_semillas_organicas' => $data['semillero_vivero/sem_semillas_organicas'] ?? null,
            'sem_proveniencia_semillas' => $data['semillero_vivero/sem_proveniencia_semillas'] ?? null,
            'variedades_semillas' => $this->extractMultiple($data, 'semillero_vivero/variedades_semillas'),
            'sem_control_plagas' => $data['semillero_vivero/sem_control_plagas'] ?? null,
            'sem_aplica_producto_quimico' => $data['semillero_vivero/sem_aplica_producto_quimico'] ?? null,
            'sem_producto_quimico' => $data['semillero_vivero/sem_producto_quimico'] ?? null,
            'sem_fecha_aplicacion_quimico' => $data['semillero_vivero/sem_fecha_aplicacion_quimico'] ?? null,
            'sem_aplica_preparado' => $data['semillero_vivero/sem_aplica_preparado'] ?? null,
            'sem_preparado' => $data['semillero_vivero/sem_preparado'] ?? null,
            'sem_fecha_aplicacion_preparado' => $data['semillero_vivero/sem_fecha_aplicacion_preparado'] ?? null,
            
            // Abonera
            'p1_abono_organico' => $data['abonera/p1_abono_organico'] ?? null,
            'p1_tipo_abono_organico' => $data['abonera/p1_tipo_abono_organico'] ?? null,
            'p1_plantas_abono_organico' => $data['abonera/p1_plantas_abono_organico'] ?? null,
            'p1_kilos_litros_abono_composta' => $data['abonera/p1_kilos_litros_abono_composta'] ?? null,
            'p1_kilos_planta_composta' => $data['abonera/p1_kilos_planta_composta'] ?? null,
            'p1_cuantos_litros_lixiviado_lombricomposta' => $data['abonera/p1_cuantos_litros_lixiviado_lombricomposta'] ?? null,
            'p1_plantas_foliar_lombricomposta' => $data['abonera/p1_plantas_foliar_lombricomposta'] ?? null,
            'p1_donde_aplico_composta' => $data['abonera/p1_donde_aplico_composta'] ?? null,
            'p1_composta_vivero' => $data['abonera/p1_composta_vivero'] ?? null,
            'p1_composta_vivero_bolsas' => $data['abonera/p1_composta_vivero_bolsas'] ?? null,
            'p1_compra_abono_organico' => $data['abonera/p1_compra_abono_organico'] ?? null,
            'p1_compra_abono_cumplimiento' => $data['abonera/p1_compra_abono_cumplimiento'] ?? null,
            'p1_cosecha_abono' => $data['abonera/p1_cosecha_abono'] ?? null,
            'p1_plantas_abonadas' => $data['abonera/p1_plantas_abonadas'] ?? null,
            'p1_aplica_estiercol_crudo' => $data['abonera/p1_aplica_estiercol_crudo'] ?? null,
            'p1_estiercol_crudo_plantas' => $data['abonera/p1_estiercol_crudo_plantas'] ?? null,
            'p1_estiercol_crudo_kg' => $data['abonera/p1_estiercol_crudo_kg'] ?? null,
            
            // Beneficio Húmedo
            'bh_fermentacion' => $data['beneficio_humedo/bh_fermentacion'] ?? null,
            'bh_fuente_agua_lavado' => $data['beneficio_humedo/bh_fuente_agua_lavado'] ?? null,
            'bh_lavado_agua_potable' => $data['beneficio_humedo/bh_lavado_agua_potable'] ?? null,
            'bh_destino_agua_miel' => $data['beneficio_humedo/bh_destino_agua_miel'] ?? null,
            'bh_despulpadora_tipo' => $data['beneficio_humedo/bh_despulpadora_tipo'] ?? null,
            'bh_despulpadora_exclusiva' => $data['beneficio_humedo/bh_despulpadora_exclusiva'] ?? null,
            'bh_destino_pulpa' => $data['beneficio_humedo/bh_destino_pulpa'] ?? null,
            'bh_secado' => $data['beneficio_humedo/bh_secado'] ?? null,
            'bh_riesgo_contaminacion' => $data['beneficio_humedo/bh_riesgo_contaminacion'] ?? null,
            'bh_riesgo_contaminacion_area' => $data['beneficio_humedo/bh_riesgo_contaminacion_area'] ?? null,
            'bh_riesgos_mezcla_producto' => $data['beneficio_humedo/bh_riesgos_mezcla_producto'] ?? null,
            'bh_instalaciones_compartidas' => $data['beneficio_humedo/bh_instalaciones_compartidas'] ?? null,
            'bh_medidas_prevencion' => $data['beneficio_humedo/bh_medidas_prevencion'] ?? null,
            
            // Almacenamiento
            'bodega_especial' => $data['almacenamiento_insumos/bodega_especial'] ?? null,
            'almacenamiento_exclusivo' => $data['almacenamiento_insumos/almacenamiento_exclusivo'] ?? null,
            'prevencion_contaminacion_almacenamiento' => $this->extractMultiple($data, 'almacenamiento_insumos/prevencion_contaminacion_almacenamiento'),
            'riesgos_mezcla_producto' => $data['almacenamiento_insumos/riesgos_mezcla_producto'] ?? null,
            'prevencion_plagas_almacen' => $data['almacenamiento_insumos/prevencion_plagas_almacen'] ?? null,
            'prevencion_plagas_almacen_describir' => $data['almacenamiento_insumos/prevencion_plagas_almacen_describir'] ?? null,
            'envases_exclusivos' => $data['almacenamiento_insumos/envases_exclusivos'] ?? null,
            'riegos_costales' => $data['almacenamiento_insumos/riegos_costales'] ?? null,
            
            // Capacitaciones
            'cap_capacitaciones' => $data['capacitaciones/cap_capacitaciones'] ?? null,
            'cap_programa_individual' => $data['capacitaciones/cap_programa_individual'] ?? null,
            'cap_registro_promotor' => $data['capacitaciones/cap_registro_promotor'] ?? null,
            'cap_capacitacion_recibida' => $data['capacitaciones/cap_capacitacion_recibida'] ?? null,
            'cap_capacitacion_fecha' => $data['capacitaciones/cap_capacitacion_fecha'] ?? null,
            'cap_capacitacion_temas' => $this->extractMultiple($data, 'capacitaciones/cap_capacitacion_temas'),
            'cap_practica_plan_manejo_organico' => $data['capacitaciones/cap_practica_plan_manejo_organico'] ?? null,
            'cap_conoce_insumos_permitidos' => $data['capacitaciones/cap_conoce_insumos_permitidos'] ?? null,
            'cap_conoce_reglamento_interno' => $data['capacitaciones/cap_conoce_reglamento_interno'] ?? null,
            'cap_recibos_ventas' => $data['capacitaciones/cap_recibos_ventas'] ?? null,
            'conocimiento_premio_social' => $data['capacitaciones/conocimiento_premio_social'] ?? null,
            'recibe_premio_social' => $data['capacitaciones/recibe_premio_social'] ?? null,
            'uso_premio_social' => $data['capacitaciones/uso_premio_social'] ?? null,
            
            // Información de Parcelas
            'n_parcelas' => $data['num_parcelas/n_parcelas'] ?? null,
            'parcela_nueva' => $data['num_parcelas/parcela_nueva'] ?? null,
            
            // Parcelas (array de hasta 9 parcelas + 1 adicional)
            'parcelas' => $this->extractAllParcelas($data),
            
            // Control de Plagas General
            'control_plagas' => $data['parcela_1/control_plagas'] ?? null,
            'insumos_plagas_seleccionados' => $this->extractMultiple($data, 'parcela_1/insumos_plagas_seleccionados'),
            'control_plagas_fecha' => $data['parcela_1/control_plagas_fecha'] ?? null,
            'insumos_aprobados' => $data['parcela_1/insumos_aprobados'] ?? null,
            'aspersora' => $data['parcela_1/aspersora'] ?? null,
            
            // Conservación
            'conservacion_agua' => $data['parcela_1/conservacion_agua'] ?? null,
            'conservacion_agua_actividades' => $this->extractMultiple($data, 'parcela_1/conservacion_agua_actividades'),
            'aplica_ceniza' => $data['parcela_1/aplica_ceniza'] ?? null,
            'ceniza_plantas' => $data['parcela_1/ceniza_plantas'] ?? null,
            'ceniza_kg' => $data['parcela_1/ceniza_kg'] ?? null,
            'ceniza_origen' => $data['parcela_1/ceniza_origen'] ?? null,
            'ceniza_tratada' => $data['parcela_1/ceniza_tratada'] ?? null,
            
            // Basura
            'basura_inorganica' => $data['parcela_1/basura_inorganica'] ?? null,
            
            // Cosecha y Comercialización
            'fecha_cosecha_organico' => $data['cosecha_acopio_comercializacion/fecha_cosecha_organico'] ?? null,
            'entrega_cafe' => $this->extractMultiple($data, 'cosecha_acopio_comercializacion/entrega_cafe'),
            'procesamiento' => $data['cosecha_acopio_comercializacion/procesamiento'] ?? null,
            'venta_otro_productor' => $data['cosecha_acopio_comercializacion/venta_otro_productor'] ?? null,
            'otro_productor' => $data['cosecha_acopio_comercializacion/otro_productor'] ?? null,
            
            // Resumen
            'parcelas_totales' => $data['resumen_parcelas/parcelas_totales'] ?? null,
            'area_total' => $data['resumen_parcelas/area_total'] ?? null,
            'plantas_totales' => $data['resumen_parcelas/plantas_totales'] ?? null,
            'plantillas_totales' => $data['resumen_parcelas/plantillas_totales'] ?? null,
            'kg_estimados_totales' => $data['resumen_parcelas/kg_estimados_totales'] ?? null,
            'kg_cosecha_pasada' => $data['resumen_parcelas/kg_cosecha_pasada'] ?? null,
            'observaciones_generales' => $data['resumen_parcelas/observaciones_generales'] ?? null,
            
            // Comprobación
            'razon_baja' => $data['comprobacion/razon_baja'] ?? null,
            'hora_final' => $data['comprobacion/hora_final'] ?? null,
            
            // Raw data completo
            'raw_data' => $data
        ];
    }

    protected function extractAllParcelas(array $data): array
    {
        $parcelas = [];
        
        // Extraer parcelas regulares (1-9)
        for ($i = 1; $i <= 9; $i++) {
            $parcelaData = $this->extractParcelaData($data, $i);
            
            // Solo agregar si tiene nombre
            if (!empty($parcelaData['nombre'])) {
                $parcelas[] = $parcelaData;
            }
        }
        
        // Extraer parcela adicional si existe
        if (isset($data['parcela_ad/pad_nombre_dis']) && !empty($data['parcela_ad/pad_nombre_dis'])) {
            $parcelas[] = $this->extractParcelaAdicional($data);
        }
        
        return $parcelas;
    }

    protected function extractParcelaData(array $data, int $numero): array
    {
        $prefix = "parcela_{$numero}";
        $p = "p{$numero}";
        
        // PREFIJOS CORRECTOS SEGÚN LOS DATOS REALES DE KOBO:
        // Parcela 1: "terrenos_colindantes_entorno" y "cosecha_acopio_comercializacion"
        // Parcela 2+: "p2_terrenos_colindantes_entorno" y "p2_cosecha_acopio_comercializacion"
        
        if ($numero === 1) {
            $prefixTerrenos = "terrenos_colindantes_entorno";
            $prefixCosecha = "cosecha_acopio_comercializacion";
        } else {
            $prefixTerrenos = "{$p}_terrenos_colindantes_entorno";
            $prefixCosecha = "{$p}_cosecha_acopio_comercializacion";
        }
        
        return [
            'numero' => $numero,
            'codigo' => $data["{$prefix}/{$p}_codigo"] ?? null,
            'nombre' => $data["{$prefix}/{$p}_nombre"] ?? null,
            'superficie' => $data["{$prefix}/{$p}_area"] ?? null,
            'activa' => $data["{$prefix}/{$p}_activa"] ?? null,
            'variacion_parcela' => $data["{$prefix}/{$p}_variacion_parcela"] ?? null,
            'area_actual' => $data["{$prefix}/{$p}_area_actual"] ?? $data["{$prefix}/{$p}_area"] ?? null,
            'variacion_explicacion' => $data["{$prefix}/{$p}_variacion_explicacion"] ?? null,
            
            // Características de la parcela
            'variedades' => $this->extractMultiple($data, "{$prefix}/{$p}_variedades"),
            'tipo_sombra' => $this->extractMultiple($data, "{$prefix}/{$p}_tipo_sombra"),
            'estado_sombra' => $data["{$prefix}/{$p}_estado_sombra"] ?? null,
            'arboles_altos' => $data["{$prefix}/{$p}_arboles_altos"] ?? null,
            'arboles_altos_numero' => $data["{$prefix}/{$p}_arboles_altos_numero"] ?? null,
            'talas_inecesaria' => $data["{$prefix}/{$p}_talas_inecesaria"] ?? null,
            'renovacion_sombras' => $data["{$prefix}/{$p}_renovacion_sombras"] ?? null,
            'colindante_talas' => $data["{$prefix}/{$p}_colindante_talas"] ?? null,
            'materia_organica' => $data["{$prefix}/{$p}_materia_organica"] ?? null,
            'cobertura' => $data["{$prefix}/{$p}_cobertura"] ?? null,
            'deslavado' => $data["{$prefix}/{$p}_deslavado"] ?? null,
            
            // Prácticas de conservación
            'practicas_conservacion' => $data["{$prefix}/{$p}_practicas_conservacion"] ?? null,
            'barreras_vivas' => $data["{$prefix}/{$p}_barreras_vivas"] ?? null,
            'metros_barreras_vivas' => $data["{$prefix}/{$p}_metros_barreras_vivas"] ?? null,
            'terrazas' => $data["{$prefix}/{$p}_terrazas"] ?? null,
            'numero_terrazas' => $data["{$prefix}/{$p}_numero_terrazas"] ?? null,
            'mantemimiento' => $data["{$prefix}/{$p}_mantemimiento"] ?? null,
            'mantenimiento_descripcion' => $data["{$prefix}/{$p}_mantenimiento_descripcion"] ?? null,
            
            // Manejo
            'aplica_estiercol_crudo' => $data["{$prefix}/{$p}_aplica_estiercol_crudo"] ?? null,
            'estiercol_crudo_plantas' => $data["{$prefix}/{$p}_estiercol_crudo_plantas"] ?? null,
            'estiercol_crudo_kg' => $data["{$prefix}/{$p}_estiercol_crudo_kg"] ?? null,
            'podas' => $data["{$prefix}/{$p}_podas"] ?? null,
            'podas_descripcion' => $this->extractMultiple($data, "{$prefix}/{$p}_podas_descripcion"),
            'plagas_enfermedades' => $this->extractMultiple($data, "{$prefix}/{$p}_plagas_enfermedades"),
            'aplica_quimico' => $data["{$prefix}/{$p}_aplica_quimico"] ?? null,
            'producto_quimico' => $data["{$prefix}/{$p}_producto_quimico"] ?? null,
            'fecha_producto_quimico' => $data["{$prefix}/{$p}_fecha_producto_quimico"] ?? null,
            'condiciones' => $data["{$prefix}/{$p}_condiciones"] ?? null,
            'basura' => $data["{$prefix}/{$p}_basura"] ?? null,
            'riesgo' => $data["{$prefix}/{$p}_riesgo"] ?? null,
            'observaciones' => $data["{$prefix}/{$p}_observaciones"] ?? null,
            
            // Terrenos colindantes - PREFIJO CORRECTO
            'norte' => $data["{$prefixTerrenos}/{$p}_norte"] ?? null,
            'norte_propietario' => $data["{$prefixTerrenos}/{$p}_norte_propietario"] ?? null,
            'sur' => $data["{$prefixTerrenos}/{$p}_sur"] ?? null,
            'sur_propietario' => $data["{$prefixTerrenos}/{$p}_sur_propietario"] ?? null,
            'este' => $data["{$prefixTerrenos}/{$p}_este"] ?? null,
            'este_propietario' => $data["{$prefixTerrenos}/{$p}_este_propietario"] ?? null,
            'oeste' => $data["{$prefixTerrenos}/{$p}_oeste"] ?? null,
            'oeste_propietario' => $data["{$prefixTerrenos}/{$p}_oeste_propietario"] ?? null,
            'herbicidas_colindantes' => $data["{$prefixTerrenos}/{$p}_herbicidas_colindantes"] ?? null,
            'nombre_herbicida_colindante' => $this->extractMultiple($data, "{$prefixTerrenos}/{$p}_nombre_herbicida_colindante"),
            'riesgos_contaminacion' => $data["{$prefixTerrenos}/{$p}_riesgos_contaminacion"] ?? null,
            'lado_riesgos_contaminacion' => $data["{$prefixTerrenos}/{$p}_lado_riesgos_contaminacion"] ?? null,
            'franja_amortiguamiento' => $data["{$prefixTerrenos}/{$p}_franja_amortiguamiento"] ?? null,
            'zanja' => $data["{$prefixTerrenos}/{$p}_zanja"] ?? null,
            'barreras_vivas_proteccion' => $data["{$prefixTerrenos}/{$p}_barreras_vivas_001"] ?? null,
            'ninguna_evita_contaminacion' => $data["{$prefixTerrenos}/{$p}_ninguna_evita_contaminacion"] ?? null,
            'glifosato' => $data["{$prefixTerrenos}/{$p}_glifosato"] ?? null,
            'aplicacion_glifosato' => $data["{$prefixTerrenos}/{$p}_aplicacion_glifosato"] ?? null,
            'aplicacion_glifosato_metros' => $data["{$prefixTerrenos}/{$p}_aplicacion_glifosato_metros"] ?? null,
            
            // Estimación de cosecha - PREFIJO CORRECTO
            'aspecto_nivel1' => $data["{$prefixCosecha}/{$p}_aspecto_nivel1"] ?? null,
            'aspecto_nivel2' => $data["{$prefixCosecha}/{$p}_aspecto_nivel2"] ?? null,
            'aspecto_nivel3' => $data["{$prefixCosecha}/{$p}_aspecto_nivel3"] ?? null,
            'n_plantas_n1' => $data["{$prefixCosecha}/{$p}_n_plantas_n1"] ?? null,
            'n_plantas_n2' => $data["{$prefixCosecha}/{$p}_n_plantas_n2"] ?? null,
            'n_plantas_n3' => $data["{$prefixCosecha}/{$p}_n_plantas_n3"] ?? null,
            'n_plantas_tot' => $data["{$prefixCosecha}/{$p}_n_plantas_tot"] ?? null,
            'edad_plantas' => $data["{$prefixCosecha}/{$p}_edad_plantas"] ?? null,
            'plantillas' => $data["{$prefixCosecha}/{$p}_plantillas"] ?? null,
            'plantillas_numero' => $data["{$prefixCosecha}/{$p}_plantillas_numero"] ?? null,
            'kg_pergamino_n1' => $data["{$prefixCosecha}/{$p}_kg_pergamino_n1"] ?? null,
            'kg_pergamino_n2' => $data["{$prefixCosecha}/{$p}_kg_pergamino_n2"] ?? null,
            'kg_pergamino_n3' => $data["{$prefixCosecha}/{$p}_kg_pergamino_n3"] ?? null,
            'kg_pergamino_tot' => $data["{$prefixCosecha}/{$p}_kg_pergamino_tot"] ?? null,
            'condiciones_productivas' => $data["{$prefixCosecha}/{$p}_condiciones_productivas"] ?? null,
        ];
    }
    public function diagnostico($id)
    {
        try {
            $inspeccion = Inspeccion::findOrFail($id);
            
            // Obtener los datos raw
            $rawData = $inspeccion->raw_data;
            
            if (!$rawData) {
                return response()->json([
                    'error' => 'No hay datos raw disponibles',
                    'inspeccion_id' => $id
                ]);
            }
            
            // Analizar campos relacionados con parcela 1
            $parcela1Fields = [];
            $terrenosFields = [];
            $cosechaFields = [];
            
            foreach ($rawData as $key => $value) {
                // Campos de parcela_1
                if (strpos($key, 'parcela_1/') === 0) {
                    $parcela1Fields[$key] = is_array($value) ? 'ARRAY' : (strlen((string)$value) > 50 ? substr($value, 0, 50) . '...' : $value);
                }
                
                // Campos de terrenos colindantes
                if (strpos($key, 'p1_terrenos_colindantes') !== false || 
                    strpos($key, 'terrenos_colindantes') !== false) {
                    $terrenosFields[$key] = is_array($value) ? 'ARRAY' : (strlen((string)$value) > 50 ? substr($value, 0, 50) . '...' : $value);
                }
                
                // Campos de cosecha
                if (strpos($key, 'p1_cosecha') !== false || 
                    strpos($key, 'cosecha_acopio') !== false) {
                    $cosechaFields[$key] = is_array($value) ? 'ARRAY' : (strlen((string)$value) > 50 ? substr($value, 0, 50) . '...' : $value);
                }
            }
            
            // Verificar cómo se guardaron las parcelas
            $parcelaGuardada = null;
            if (isset($inspeccion->parcelas) && is_array($inspeccion->parcelas) && count($inspeccion->parcelas) > 0) {
                $parcelaGuardada = $inspeccion->parcelas[0];
            }
            
            return response()->json([
                'inspeccion_id' => $id,
                'total_campos_raw' => count($rawData),
                'campos_parcela_1' => [
                    'total' => count($parcela1Fields),
                    'campos' => $parcela1Fields
                ],
                'campos_terrenos_colindantes' => [
                    'total' => count($terrenosFields),
                    'campos' => $terrenosFields
                ],
                'campos_cosecha' => [
                    'total' => count($cosechaFields),
                    'campos' => $cosechaFields
                ],
                'parcela_guardada' => $parcelaGuardada,
                'tiene_observaciones' => isset($parcelaGuardada['observaciones']),
                'tiene_norte' => isset($parcelaGuardada['norte']),
                'tiene_kg_tot' => isset($parcelaGuardada['kg_pergamino_tot']),
                
                // Buscar patrones específicos
                'patrones_encontrados' => [
                    'parcela_1/' => $this->contarPatron($rawData, 'parcela_1/'),
                    'p1_' => $this->contarPatron($rawData, 'p1_'),
                    '_terrenos_' => $this->contarPatron($rawData, '_terrenos_'),
                    '_cosecha_' => $this->contarPatron($rawData, '_cosecha_'),
                    '_entorno' => $this->contarPatron($rawData, '_entorno'),
                    '_comercializacion' => $this->contarPatron($rawData, '_comercializacion'),
                ]
            ], JSON_PRETTY_PRINT);
            
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ], 500);
        }
    }
    
    private function contarPatron($data, $patron)
    {
        $count = 0;
        $ejemplos = [];
        foreach ($data as $key => $value) {
            if (stripos($key, $patron) !== false) {
                $count++;
                if (count($ejemplos) < 5) {
                    $ejemplos[] = $key;
                }
            }
        }
        return [
            'cantidad' => $count,
            'ejemplos' => $ejemplos
        ];
    }
    
    /**
     * Método alternativo para extraer datos de parcelas si los prefijos no coinciden
     */
    protected function extractParcelaDataFlexible(array $data, int $numero): array
    {
        $p = "p{$numero}";
        
        // Buscar todos los posibles prefijos
        $posiblesPrefijos = [
            "parcela_{$numero}",
            "parcela{$numero}",
            $p
        ];
        
        $posiblesTerrenos = [
            "{$p}_terrenos_colindantes_entorno",
            "p{$numero}_terrenos_colindantes",
            "terrenos_colindantes_{$p}",
            "parcela_{$numero}_terrenos"
        ];
        
        $posiblesCosecha = [
            "{$p}_cosecha_acopio_comercializacion",
            "p{$numero}_cosecha_acopio",
            "cosecha_{$p}",
            "parcela_{$numero}_cosecha"
        ];
        
        $result = [
            'numero' => $numero,
            'codigo' => null,
            'nombre' => null,
            // ... todos los demás campos inicializados en null
        ];
        
        // Función helper para buscar con múltiples prefijos
        $buscarCampo = function($sufijo, $prefijos) use ($data) {
            foreach ($prefijos as $prefijo) {
                $key = "{$prefijo}/{$sufijo}";
                if (isset($data[$key])) {
                    return $data[$key];
                }
                // Intentar sin la barra
                $key2 = "{$prefijo}_{$sufijo}";
                if (isset($data[$key2])) {
                    return $data[$key2];
                }
            }
            return null;
        };
        
        // Extraer campos básicos
        $result['codigo'] = $buscarCampo("{$p}_codigo", $posiblesPrefijos);
        $result['nombre'] = $buscarCampo("{$p}_nombre", $posiblesPrefijos);
        $result['superficie'] = $buscarCampo("{$p}_area", $posiblesPrefijos);
        
        // Extraer terrenos colindantes
        $result['norte'] = $buscarCampo("{$p}_norte", $posiblesTerrenos);
        $result['norte_propietario'] = $buscarCampo("{$p}_norte_propietario", $posiblesTerrenos);
        
        // Extraer cosecha
        $result['kg_pergamino_tot'] = $buscarCampo("{$p}_kg_pergamino_tot", $posiblesCosecha);
        $result['n_plantas_tot'] = $buscarCampo("{$p}_n_plantas_tot", $posiblesCosecha);
        
        return $result;
    }

    protected function extractParcelaAdicional(array $data): array
    {
        return [
            'numero' => 'adicional',
            'codigo' => $data['parcela_ad/pad_cod'] ?? null,
            'nombre' => $data['parcela_ad/pad_nombre_dis'] ?? null,
            'superficie' => $data['parcela_ad/pad_area'] ?? null,
            'numero_parcela_adicional' => $data['parcela_ad/numero_p_adicional'] ?? null,
            
            // Características
            'variedades' => $this->extractMultiple($data, 'parcela_ad/pad_variedades'),
            'tipo_sombra' => $this->extractMultiple($data, 'parcela_ad/pad_tipo_sombra'),
            'estado_sombra' => $data['parcela_ad/pad_estado_sombra'] ?? null,
            'materia_organica' => $data['parcela_ad/pad_materia_organica'] ?? null,
            'cobertura' => $data['parcela_ad/pad_cobertura'] ?? null,
            'deslavado' => $data['parcela_ad/pad_deslavado'] ?? null,
            
            // Prácticas
            'practicas_conservacion' => $data['parcela_ad/pad_practicas_conservacion'] ?? null,
            'barreras_vivas' => $data['parcela_ad/pad_barreras_vivas'] ?? null,
            'metros_barreras_vivas' => $data['parcela_ad/pad_metros_barreras_vivas'] ?? null,
            'terrazas' => $data['parcela_ad/pad_terrazas'] ?? null,
            'numero_terrazas' => $data['parcela_ad/pad_numero_terrazas'] ?? null,
            'mantemimiento' => $data['parcela_ad/pad_mantemimiento'] ?? null,
            'mantenimiento_descripcion' => $data['parcela_ad/pad_mantenimiento_descripcion'] ?? null,
            
            // Manejo
            'aplica_estiercol_crudo' => $data['parcela_ad/pad_aplica_estiercol_crudo'] ?? null,
            'estiercol_crudo_plantas' => $data['parcela_ad/pad_estiercol_crudo_plantas'] ?? null,
            'estiercol_crudo_kg' => $data['parcela_ad/pad_estiercol_crudo_kg'] ?? null,
            'podas' => $data['parcela_ad/pad_podas'] ?? null,
            'podas_descripcion' => $this->extractMultiple($data, 'parcela_ad/pad_podas_descripcion'),
            'plagas_enfermedades' => $this->extractMultiple($data, 'parcela_ad/pad_plagas_enfermedades'),
            'aplica_quimico' => $data['parcela_ad/pad_aplica_quimico'] ?? null,
            'producto_quimico' => $data['parcela_ad/pad_producto_quimico'] ?? null,
            'fecha_producto_quimico' => $data['parcela_ad/pad_fecha_producto_quimico'] ?? null,
            'condiciones' => $data['parcela_ad/pad_condiciones'] ?? null,
            'basura' => $data['parcela_ad/pad_basura'] ?? null,
            'riesgo' => $data['parcela_ad/pad_riesgo'] ?? null,
            'observaciones' => $data['parcela_ad/pad_observaciones'] ?? null,
            
            // Colindantes
            'norte' => $data['pad_terrenos_colindantes_entorno/pad_norte'] ?? null,
            'norte_propietario' => $data['pad_terrenos_colindantes_entorno/pad_norte_propietario'] ?? null,
            'sur' => $data['pad_terrenos_colindantes_entorno/pad_sur'] ?? null,
            'sur_propietario' => $data['pad_terrenos_colindantes_entorno/pad_sur_propietario'] ?? null,
            'este' => $data['pad_terrenos_colindantes_entorno/pad_este'] ?? null,
            'este_propietario' => $data['pad_terrenos_colindantes_entorno/pad_este_propietario'] ?? null,
            'oeste' => $data['pad_terrenos_colindantes_entorno/pad_oeste'] ?? null,
            'oeste_propietario' => $data['pad_terrenos_colindantes_entorno/pad_oeste_propietario'] ?? null,
            'herbicidas_colindantes' => $data['pad_terrenos_colindantes_entorno/pad_herbicidas_colindantes'] ?? null,
            'nombre_herbicida_colindante' => $data['pad_terrenos_colindantes_entorno/pad_nombre_herbicida_colindante'] ?? null,
            'riesgos_contaminacion' => $data['pad_terrenos_colindantes_entorno/pad_riesgos_contaminacion'] ?? null,
            'lado_riesgos_contaminacion' => $data['pad_terrenos_colindantes_entorno/pad_lado_riesgos_contaminacion'] ?? null,
            'franja_amortiguamiento' => $data['pad_terrenos_colindantes_entorno/pad_franja_amortiguamiento'] ?? null,
            'zanja' => $data['pad_terrenos_colindantes_entorno/pad_zanja'] ?? null,
            'barreras_vivas_proteccion' => $data['pad_terrenos_colindantes_entorno/pad_barreras_vivas_001'] ?? null,
            'ninguna_evita_contaminacion' => $data['pad_terrenos_colindantes_entorno/pad_ninguna_evita_contaminacion'] ?? null,
            'glifosato' => $data['pad_terrenos_colindantes_entorno/pad_glifosato'] ?? null,
            'aplicacion_glifosato' => $data['pad_terrenos_colindantes_entorno/pad_aplicacion_glifosato'] ?? null,
            'aplicacion_glifosato_metros' => $data['pad_terrenos_colindantes_entorno/pad_aplicacion_glifosato_metros'] ?? null,
            
            // Estimación
            'aspecto_nivel1' => $data['pad_cosecha_acopio_comercializacion/pad_aspecto_nivel1'] ?? null,
            'aspecto_nivel2' => $data['pad_cosecha_acopio_comercializacion/pad_aspecto_nivelad'] ?? null,
            'aspecto_nivel3' => $data['pad_cosecha_acopio_comercializacion/pad_aspecto_nivel3'] ?? null,
            'n_plantas_n1' => $data['pad_cosecha_acopio_comercializacion/pad_n_plantas_n1'] ?? null,
            'n_plantas_n2' => $data['pad_cosecha_acopio_comercializacion/pad_n_plantas_n2'] ?? null,
            'n_plantas_n3' => $data['pad_cosecha_acopio_comercializacion/pad_n_plantas_n3'] ?? null,
            'n_plantas_tot' => $data['pad_cosecha_acopio_comercializacion/pad_n_plantas_tot'] ?? null,
            'edad_plantas' => $data['pad_cosecha_acopio_comercializacion/pad_edad_plantas'] ?? null,
            'plantillas' => $data['pad_cosecha_acopio_comercializacion/pad_plantillas'] ?? null,
            'plantillas_numero' => $data['pad_cosecha_acopio_comercializacion/pad_plantillas_numero'] ?? null,
            'kg_pergamino_n1' => $data['pad_cosecha_acopio_comercializacion/pad_kg_pergamino_n1'] ?? null,
            'kg_pergamino_n2' => $data['pad_cosecha_acopio_comercializacion/pad_kg_pergamino_n2'] ?? null,
            'kg_pergamino_n3' => $data['pad_cosecha_acopio_comercializacion/pad_kg_pergamino_n3'] ?? null,
            'kg_pergamino_tot' => $data['pad_cosecha_acopio_comercializacion/pad_kg_pergamino_tot'] ?? null,
        ];
    }

    protected function extractMultiple(array $data, string $key): ?array
    {
        if (!isset($data[$key])) {
            return null;
        }
        
        $value = $data[$key];
        
        if (is_array($value)) {
            return $value;
        }
        
        if (is_string($value) && str_contains($value, ' ')) {
            return array_filter(explode(' ', $value));
        }
        
        return [$value];
    }

    public function getData(Request $request)
    {
        try {
            $query = Inspeccion::query();

            if ($request->has('year') && $request->year) {
                $query->whereYear('fecha_visita', $request->year);
            }

            if ($request->has('inspector') && $request->inspector) {
                $query->where('inspector', $request->inspector);
            }

            if ($request->has('municipio') && $request->municipio) {
                $query->where('municipio', $request->municipio);
            }

            if ($request->has('estatus') && $request->estatus) {
                $query->where('estatus', $request->estatus);
            }

            $inspecciones = $query->orderBy('fecha_visita', 'desc')->get();

            $data = [];
            foreach ($inspecciones as $inspeccion) {
                if (!empty($inspeccion->parcelas)) {
                    foreach ($inspeccion->parcelas as $parcela) {
                        $data[] = $this->formatRowData($inspeccion, $parcela);
                    }
                } else {
                    $data[] = $this->formatRowData($inspeccion, null);
                }
            }

            return response()->json(['data' => $data]);

        } catch (\Exception $e) {
            Log::error('Error obteniendo datos: ' . $e->getMessage());
            
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }



    protected function formatRowData($inspeccion, $parcela = null): array
    {
        $row = [
            // Meta (0-2)
            $inspeccion->kobo_id ?? '',
            $inspeccion->created_at ? $inspeccion->created_at->format('Y-m-d H:i') : '',
            $inspeccion->submission_time ?? '',
            
            // Info General (3-9)
            $inspeccion->fecha_visita ?? '',
            $inspeccion->hora_inicio ?? '',
            $inspeccion->inspector ?? '',
            $inspeccion->gps_productor ?? '',
            $inspeccion->productor_nuevo ?? '',
            $inspeccion->otra_organizacion ?? '',
            $inspeccion->nombre_otra_organizacion ?? '',
            
            // Productor (10-37)
            $inspeccion->dni ?? '',
            $inspeccion->id_inspeccion ?? '',
            $inspeccion->socio ?? '',
            $inspeccion->curp ?? '',
            $inspeccion->rfc ?? '',
            $inspeccion->ine ?? '',
            $inspeccion->folio_cafetalero ?? '',
            $inspeccion->folio_suri ?? '',
            $inspeccion->municipio ?? '',
            $inspeccion->localidad ?? '',
            $inspeccion->grupo_trabajo ?? '',
            $inspeccion->telefono ?? '',
            $inspeccion->genero ?? '',
            $inspeccion->fecha_nacimiento ?? '',
            $inspeccion->fecha_ingreso ?? '',
            $inspeccion->estatus ?? '',
            $inspeccion->fecha_transicion ?? '',
            $inspeccion->baja_productor ?? '',
            is_array($inspeccion->tenencia_tierra) ? implode(', ', $inspeccion->tenencia_tierra) : ($inspeccion->tenencia_tierra ?? ''),
            $inspeccion->titulo_tenencia ?? '',
            $inspeccion->papel_tenencia ?? '',
            $inspeccion->participa_otra_organizacion ?? '',
            $inspeccion->producto_otra_organizacion ?? '',
            $inspeccion->nombre_otra_organizacion_001 ?? '',
            $inspeccion->familia_participa_otra_organizacion ?? '',
            $inspeccion->familia_producto_otra_organizazacion ?? '',
            $inspeccion->familia_otra_organizacion ?? '',
            $inspeccion->familia_categoria_otra_organizacion ?? '',
            
            // Área General (38-51)
            $inspeccion->tiene_area_autoconsumo ?? '',
            $inspeccion->area_autoconsumo ?? '',
            $inspeccion->autoconsumo_n_parcelas ?? '',
            $inspeccion->autoconsumo_quimicos ?? '',
            $inspeccion->autoconsumo_fecha_quimicos ?? '',
            $inspeccion->tiene_area_potrero ?? '',
            $inspeccion->area_potrero ?? '',
            $inspeccion->potrero_n_parcelas ?? '',
            $inspeccion->tiene_area_bosque ?? '',
            $inspeccion->area_bosque ?? '',
            $inspeccion->bosque_n_parcelas ?? '',
            $inspeccion->tiene_area_acahual ?? '',
            $inspeccion->area_acahual ?? '',
            $inspeccion->acahual_n_parcelas ?? '',
            
            // Semillero (52-63)
            $inspeccion->sem_semillero ?? '',
            $inspeccion->sem_numero_semilleros ?? '',
            $inspeccion->sem_semillas_organicas ?? '',
            $inspeccion->sem_proveniencia_semillas ?? '',
            is_array($inspeccion->variedades_semillas) ? implode(', ', $inspeccion->variedades_semillas) : ($inspeccion->variedades_semillas ?? ''),
            $inspeccion->sem_control_plagas ?? '',
            $inspeccion->sem_aplica_producto_quimico ?? '',
            $inspeccion->sem_producto_quimico ?? '',
            $inspeccion->sem_fecha_aplicacion_quimico ?? '',
            $inspeccion->sem_aplica_preparado ?? '',
            $inspeccion->sem_preparado ?? '',
            $inspeccion->sem_fecha_aplicacion_preparado ?? '',
            
            // Abonera (64-80)
            $inspeccion->p1_abono_organico ?? '',
            $inspeccion->p1_tipo_abono_organico ?? '',
            is_array($inspeccion->p1_plantas_abono_organico) ? implode(', ', $inspeccion->p1_plantas_abono_organico) : ($inspeccion->p1_plantas_abono_organico ?? ''),
            $inspeccion->p1_kilos_litros_abono_composta ?? '',
            $inspeccion->p1_kilos_planta_composta ?? '',
            $inspeccion->p1_cuantos_litros_lixiviado_lombricomposta ?? '',
            $inspeccion->p1_plantas_foliar_lombricomposta ?? '',
            $inspeccion->p1_donde_aplico_composta ?? '',
            $inspeccion->p1_composta_vivero ?? '',
            $inspeccion->p1_composta_vivero_bolsas ?? '',
            $inspeccion->p1_compra_abono_organico ?? '',
            $inspeccion->p1_compra_abono_cumplimiento ?? '',
            $inspeccion->p1_cosecha_abono ?? '',
            $inspeccion->p1_plantas_abonadas ?? '',
            $inspeccion->p1_aplica_estiercol_crudo ?? '',
            $inspeccion->p1_estiercol_crudo_plantas ?? '',
            $inspeccion->p1_estiercol_crudo_kg ?? '',
            
            // Beneficio Húmedo (81-93)
            $inspeccion->bh_fermentacion ?? '',
            $inspeccion->bh_fuente_agua_lavado ?? '',
            $inspeccion->bh_lavado_agua_potable ?? '',
            $inspeccion->bh_destino_agua_miel ?? '',
            $inspeccion->bh_despulpadora_tipo ?? '',
            $inspeccion->bh_despulpadora_exclusiva ?? '',
            $inspeccion->bh_destino_pulpa ?? '',
            $inspeccion->bh_secado ?? '',
            $inspeccion->bh_riesgo_contaminacion ?? '',
            $inspeccion->bh_riesgo_contaminacion_area ?? '',
            $inspeccion->bh_riesgos_mezcla_producto ?? '',
            $inspeccion->bh_instalaciones_compartidas ?? '',
            $inspeccion->bh_medidas_prevencion ?? '',
            
            // Almacenamiento (94-101)
            $inspeccion->bodega_especial ?? '',
            $inspeccion->almacenamiento_exclusivo ?? '',
            is_array($inspeccion->prevencion_contaminacion_almacenamiento) ? implode(', ', $inspeccion->prevencion_contaminacion_almacenamiento) : ($inspeccion->prevencion_contaminacion_almacenamiento ?? ''),
            $inspeccion->riesgos_mezcla_producto ?? '',
            $inspeccion->prevencion_plagas_almacen ?? '',
            $inspeccion->prevencion_plagas_almacen_describir ?? '',
            $inspeccion->envases_exclusivos ?? '',
            $inspeccion->riegos_costales ?? '',
            
            // Capacitaciones (102-114)
            $inspeccion->cap_capacitaciones ?? '',
            $inspeccion->cap_programa_individual ?? '',
            $inspeccion->cap_registro_promotor ?? '',
            $inspeccion->cap_capacitacion_recibida ?? '',
            $inspeccion->cap_capacitacion_fecha ?? '',
            is_array($inspeccion->cap_capacitacion_temas) ? implode(', ', $inspeccion->cap_capacitacion_temas) : ($inspeccion->cap_capacitacion_temas ?? ''),
            $inspeccion->cap_practica_plan_manejo_organico ?? '',
            $inspeccion->cap_conoce_insumos_permitidos ?? '',
            $inspeccion->cap_conoce_reglamento_interno ?? '',
            $inspeccion->cap_recibos_ventas ?? '',
            $inspeccion->conocimiento_premio_social ?? '',
            $inspeccion->recibe_premio_social ?? '',
            $inspeccion->uso_premio_social ?? '',
        ];
        
        // Agregar datos de parcela si existe (115-187)
        if ($parcela) {
            $row = array_merge($row, [
                $parcela['numero'] ?? '',
                $parcela['codigo'] ?? '',
                $parcela['nombre'] ?? '',
                $parcela['superficie'] ?? '',
                $parcela['activa'] === '0' ? 'Activa' : 'Inactiva',
                $parcela['variacion_parcela'] ?? '',
                $parcela['area_actual'] ?? '',
                $parcela['variacion_explicacion'] ?? '',
                is_array($parcela['variedades'] ?? null) ? implode(', ', $parcela['variedades']) : ($parcela['variedades'] ?? ''),
                is_array($parcela['tipo_sombra'] ?? null) ? implode(', ', $parcela['tipo_sombra']) : ($parcela['tipo_sombra'] ?? ''),
                $parcela['estado_sombra'] ?? '',
                $parcela['arboles_altos'] ?? '',
                $parcela['arboles_altos_numero'] ?? '',
                $parcela['talas_inecesaria'] ?? '',
                $parcela['renovacion_sombras'] ?? '',
                $parcela['colindante_talas'] ?? '',
                $parcela['materia_organica'] ?? '',
                $parcela['cobertura'] ?? '',
                $parcela['deslavado'] ?? '',
                $parcela['practicas_conservacion'] ?? '',
                $parcela['barreras_vivas'] ?? '',
                $parcela['metros_barreras_vivas'] ?? '',
                $parcela['terrazas'] ?? '',
                $parcela['numero_terrazas'] ?? '',
                $parcela['mantemimiento'] ?? '',
                $parcela['mantenimiento_descripcion'] ?? '',
                $parcela['aplica_estiercol_crudo'] ?? '',
                $parcela['estiercol_crudo_plantas'] ?? '',
                $parcela['estiercol_crudo_kg'] ?? '',
                $parcela['podas'] ?? '',
                is_array($parcela['podas_descripcion'] ?? null) ? implode(', ', $parcela['podas_descripcion']) : ($parcela['podas_descripcion'] ?? ''),
                is_array($parcela['plagas_enfermedades'] ?? null) ? implode(', ', $parcela['plagas_enfermedades']) : ($parcela['plagas_enfermedades'] ?? ''),
                $parcela['aplica_quimico'] ?? '',
                $parcela['producto_quimico'] ?? '',
                $parcela['fecha_producto_quimico'] ?? '',
                $parcela['condiciones'] ?? '',
                $parcela['basura'] ?? '',
                $parcela['riesgo'] ?? '',
                $parcela['observaciones'] ?? '',
                $parcela['norte'] ?? '',
                $parcela['norte_propietario'] ?? '',
                $parcela['sur'] ?? '',
                $parcela['sur_propietario'] ?? '',
                $parcela['este'] ?? '',
                $parcela['este_propietario'] ?? '',
                $parcela['oeste'] ?? '',
                $parcela['oeste_propietario'] ?? '',
                $parcela['herbicidas_colindantes'] ?? '',
                is_array($parcela['nombre_herbicida_colindante'] ?? null) ? implode(', ', $parcela['nombre_herbicida_colindante']) : ($parcela['nombre_herbicida_colindante'] ?? ''),
                $parcela['riesgos_contaminacion'] ?? '',
                $parcela['lado_riesgos_contaminacion'] ?? '',
                $parcela['franja_amortiguamiento'] ?? '',
                $parcela['zanja'] ?? '',
                $parcela['barreras_vivas_proteccion'] ?? '',
                $parcela['ninguna_evita_contaminacion'] ?? '',
                $parcela['glifosato'] ?? '',
                $parcela['aplicacion_glifosato'] ?? '',
                $parcela['aplicacion_glifosato_metros'] ?? '',
                $parcela['aspecto_nivel1'] ?? '',
                $parcela['aspecto_nivel2'] ?? '',
                $parcela['aspecto_nivel3'] ?? '',
                $parcela['n_plantas_n1'] ?? '',
                $parcela['n_plantas_n2'] ?? '',
                $parcela['n_plantas_n3'] ?? '',
                $parcela['n_plantas_tot'] ?? '',
                $parcela['edad_plantas'] ?? '',
                $parcela['plantillas'] ?? '',
                $parcela['plantillas_numero'] ?? '',
                $parcela['kg_pergamino_n1'] ?? '',
                $parcela['kg_pergamino_n2'] ?? '',
                $parcela['kg_pergamino_n3'] ?? '',
                $parcela['kg_pergamino_tot'] ?? '',
                $parcela['condiciones_productivas'] ?? '',
            ]);
        } else {
            // Llenar con valores vacíos si no hay parcela (73 campos)
            for ($i = 0; $i < 73; $i++) {
                $row[] = '';
            }
        }
        
        // Resumen (188-194)
        $row = array_merge($row, [
            $inspeccion->parcelas_totales ?? '',
            $inspeccion->area_total ?? '',
            $inspeccion->plantas_totales ?? '',
            $inspeccion->plantillas_totales ?? '',
            $inspeccion->kg_estimados_totales ?? '',
            $inspeccion->kg_cosecha_pasada ?? '',
            $inspeccion->observaciones_generales ?? '',
        ]);
        
        // Comprobación (195-196)
        $row = array_merge($row, [
            $inspeccion->razon_baja ?? '',
            $inspeccion->hora_final ?? '',
        ]);
        
        return $row;
    }
    //Formato de formulario
    public function vistaIndividual()
    {
        return view('InspeccionInterna.vista_individual');
    }

    /**
     * Obtener lista de productores con inspecciones
     */
    public function getProductores()
    {
        try {
            $productores = Inspeccion::select('dni', 'socio')
                ->whereNotNull('dni')
                ->whereNotNull('socio')
                ->groupBy('dni', 'socio')
                ->orderBy('socio')
                ->get()
                ->map(function($item) {
                    return [
                        'dni' => $item->dni,
                        'nombre' => $item->socio
                    ];
                });

            return response()->json($productores);
        } catch (\Exception $e) {
            Log::error('Error obteniendo productores: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Obtener fechas de inspección de un productor
     */
    public function getFechasInspeccion($dni)
    {
        try {
            $inspecciones = Inspeccion::where('dni', $dni)
                ->orderBy('fecha_visita', 'desc')
                ->get()
                ->map(function($item) {
                    return [
                        'id' => $item->_id,
                        'fecha' => $item->fecha_visita ? date('d/m/Y', strtotime($item->fecha_visita)) : 'Sin fecha',
                        'inspector' => $item->inspector ?? 'Sin inspector'
                    ];
                });

            return response()->json($inspecciones);
        } catch (\Exception $e) {
            Log::error('Error obteniendo fechas: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Ver inspección completa
     */
    public function verInspeccion($id)
    {
        try {
            $inspeccion = Inspeccion::findOrFail($id);
            
            // Convertir a array para facilitar el manejo en el frontend
            $data = $inspeccion->toArray();
            
            // Asegurar que parcelas sea un array
            if (!isset($data['parcelas']) || !is_array($data['parcelas'])) {
                $data['parcelas'] = [];
            }
            
            return response()->json($data);
        } catch (\Exception $e) {
            Log::error('Error obteniendo inspección: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Exportar inspección a PDF
     */
    public function exportarPDF($id)
    {
        try {
            $inspeccion = Inspeccion::findOrFail($id);
            
            // Aquí puedes usar una librería como DomPDF o mPDF
            // Por ahora retornamos los datos para que el frontend lo maneje
            
            return response()->json([
                'success' => true,
                'message' => 'Funcionalidad de PDF en desarrollo',
                'data' => $inspeccion
            ]);
            
        } catch (\Exception $e) {
            Log::error('Error exportando PDF: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
}