<?php

namespace App\Http\Controllers;

use App\Models\InternalInspection;
use Illuminate\Http\Request;
use App\Services\ImportadorParcelas;
use App\Services\ImportadorInspeccion;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class InternalInspectionController extends Controller
{
    public function index()
    {
        $currentYear = date('Y');
        $totalFormsInspeccion = InternalInspection::Where('version', $currentYear)->orWhere('version', $currentYear)->count();

        return view('InspeccionInterna.index');
    }

    public function importarKobo(Request $request)
    {
        $validator = Validator::make($request->all(),[
            'year' => 'required|integer|min:2020|max:' . (date('Y') + 1),
            'auth_method' => 'required|in:basic,token',
            'auth_credential' => 'required|string',
            'second_auth_credential' => 'required_if:auth_method,basic|nullable|string',
            'asset_id' => 'required|string',
            'server_url' => 'required|url'
        ]);
        if ($validator->fails()){
            return response()->json([
                'success' => false,
                'message' => 'Datos de entrada inválidos',
                'errors' => $validator->errors()
            ], 422);
        }

        try{
            $data = $validator->validated();

            //Crear instancia del importador
            $importador = new ImportadorInspeccion(
                year: $data['year'],
                assetId: $data['asset_id'],
                username: $data['auth_credential'],
                password: $data['second_auth_credential'] ?? null,
                serverUrl: $data['server_url']
            );

            //Ejecutamos la importación
            $resultado = $importador->importar();

            

            if ($resultado['success']){
                Log::info('Importación exitosa desde KoBoToolbox', [
                    'year' => $data['year'],
                    'total_registros' => $resultado['total']
                ]);
            }
            return response()->json($resultado);

        }catch(\Exception $e){
            Log::error('Error en importación KoBoToolbox',[
                'error' => $e->getMessage(),
                'year' => $request->year
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error interno del servidor: ' . $e->getMessage()
            ], 500);
        }
    }

    public function dataTables()
    {
        $Inspection = InternalInspection::paginate(50);

        return response()->json($Inspection);
    }

    public function infoAsset (Request $request)
    {
        $validator = Validator::make($request->all(),[
            'auth_method' => 'required|in:basic,token',
            'auth_credential' => 'required|string',
            'second_auth_credential' => 'required_if:auth_method,basic|nullable|string',
            'asset_id' => 'required|string',
            'server_url' => 'required|url',
            'year' => 'required|integer|min:2020'
        ]);
        if ($validator->fails()){
            return response()->json([
                'success' => false,
                'message' => 'Datos de entrada inválidos',
                'error' => $validator->errors()
            ], 422);
        }

        try{
            $data = $validator->validated();

            $importador = new ImportadorInspeccion(
                year: $data['year'],
                assetId: $data['asset_id'],
                username: $data['auth_credential'],
                password: $data['second_auth_credential'] ?? null,
                serverUrl:$data['server_url']
            );
            

            $info = $importador->obtenerInfoAsset();
            return response()->json($info);
        }catch(\Exception $e){
            return response()->json([
                'success'=> false,
                'message'=> 'Error al obtener la información: ' . $e->getMessage()
            ], 500);
        }
    }

     public function exportar(Request $request)
    {
        $year = $request->get('year', date('Y'));
        
        $productores = InternalInspection::where('version', $year)->get();
        
        $filename = "productores_{$year}_" . date('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($productores) {
            $file = fopen('php://output', 'w');
            
            // Headers del CSV
            fputcsv($file, [
                'ID', 'ID Productor', 'Localidad', 'Municipio', 'Región', 
                'Socio', 'Grupo Trabajo', 'CURP', 'RFC', 'INE',
                'Folio Cafetalero', 'Folio SURI', 'Fecha Nacimiento', 
                'Género', 'Estatus', 'Activo', 'Número Parcelas',
                'Fecha Importación', 'Fecha Envío'
            ]);
            
            // Datos
            foreach ($productores as $productor) {
                fputcsv($file, [
                    $productor->id,
                    $productor->id_productor,
                    $productor->localidad,
                    $productor->municipio,
                    $productor->region,
                    $productor->socio,
                    $productor->grupo_trabajo,
                    $productor->curp,
                    $productor->rfc,
                    $productor->ine,
                    $productor->folio_cafetalero,
                    $productor->folio_suri,
                    $productor->fecha_nacimiento,
                    $productor->genero,
                    $productor->estatus,
                    $productor->activo ? 'Sí' : 'No',
                    $productor->numero_parcelas,
                    $productor->fecha_importacion,
                    $productor->_submission_time
                ]);
            }
            
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    public function sincronizarAutomatico()
    {
        $config = [
            'year' => date('Y'),
            'username' => config('services.kobotoolbox.username'),
            'password' => config('services.kobotoolbox.password'),
            'asset_id' => config('services.kobotoolbox.asset_id'),
            'server_url' => config('services.kobotoolbox.server_url', 'https://eu.kobotoolbox.org')
        ];

        if (!$config['username'] || !$config['password'] || !$config['asset_id']) {
            Log::error('Configuración de KoBoToolbox incompleta para sincronización automática');
            return response()->json([
                'success' => false,
                'message' => 'Configuración incompleta'
            ]);
        }

        try {
            $importador = new ImportadorInspeccion(
                $config['year'],
                $config['username'],
                $config['password'],
                $config['asset_id'],
                $config['server_url']
            );

            $resultado = $importador->importar();
            
            Log::info('Sincronización automática completada', $resultado);
            
            return response()->json($resultado);

        } catch (\Exception $e) {
            Log::error("Error en sincronización automática: {$e->getMessage()}");
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    protected function validarProductorExiste($codigoSocio): bool 
    { 
        if(empty($codigoSocio)){ return false; } 
        return \App\Models\Producer::where('id_productor', $codigoSocio)->exists(); 
    }
}
