<?php

namespace App\Http\Controllers;

use App\Models\CoffeeParcelRegistry;
use App\Models\Producer;
use App\Services\ImportadorParcelas;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ParcelController extends Controller
{
    public function index()
    {
        $currentYear = date('Y');

        $totalParcel = CoffeeParcelRegistry::Where('version', $currentYear)->orWhere('version', $currentYear)->count();

        // $parcelasActivas = Producer::Where('activo', 1)->count();

        $parcelasActivas = CoffeeParcelRegistry::whereHas('productor', function($query) {
            $query->where('activo', 1); // solo productores activos
        })->count();


        $Parcels = CoffeeParcelRegistry::all();

        
        if ($Parcels->isEmpty()){
            $mesaje = [
                'tipo' => 'info',
                'texto' => 'No hay parcelas registradas, Importar Parcelas'
            ];
        }
        
        return view('parcelas.index', compact('totalParcel','Parcels','parcelasActivas'));
    }

    public function importarKobo(Request $request)
    {
        $validator = Validator::make($request->all(),[
            'year' => 'required|integer|min:2020|max:' . (date('Y') + 1),
            'auth_method' => 'required|in:basic,token',
            'auth_credential' => 'required|string',
            'second_auth_credential' => 'required_if:auth_method,basic|nullable|string',
            'asset_id' => 'required|string',
            'server_url' => 'required|url'
        ]);
        Log::info("Imporaciones exitosa");

        if ($validator->fails()){
            return response()->json([
                'success' => false,
                'message' => 'Datos de entrada inválidos',
                'errors' => $validator->errors()
            ], 422);
        }

        try{
            $data = $validator->validated();

            //Crear instancia del importador
            $importador = new ImportadorParcelas(
                year: $data['year'],
                assetId: $data['asset_id'],
                username: $data['auth_credential'],
                password: $data['second_auth_credential'] ?? null,
                serverUrl: $data['server_url']
            );

            //Ejecutamos la importación
            $resultado = $importador->importar();

            

            if ($resultado['success']){
                Log::info('Importación exitosa desde KoBoToolbox', [
                    'year' => $data['year'],
                    'total_registros' => $resultado['total']
                ]);
            }
            return response()->json($resultado);

        }catch(\Exception $e){
            Log::error('Error en importación KoBoToolbox',[
                'error' => $e->getMessage(),
                'year' => $request->year
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error interno del servidor: ' . $e->getMessage()
            ], 500);
        }
    }

    // Obtenemos información del asset antes de importar
    public function infoAsset (Request $request)
    {
        $validator = Validator::make($request->all(),[
            'auth_method' => 'required|in:basic,token',
            'auth_credential' => 'required|string',
            'second_auth_credential' => 'required_if:auth_method,basic|nullable|string',
            'asset_id' => 'required|string',
            'server_url' => 'required|url',
            'year' => 'required|integer|min:2020'
        ]);

        if ($validator->fails()){
            return response()->json([
                'success' => false,
                'message' => 'Datos de entrada inválidos',
                'error' => $validator->errors()
            ], 422);
        }

        try{
            $data = $validator->validated();

            $importador = new ImportadorParcelas(
                year: $data['year'],
                assetId: $data['asset_id'],
                username: $data['auth_credential'],
                password: $data['second_auth_credential'] ?? null,
                serverUrl:$data['server_url']
            );
            

            $info = $importador->obtenerInfoAsset();
            return response()->json($info);
        }catch(\Exception $e){
            return response()->json([
                'success'=> false,
                'message'=> 'Error al obtener la información: ' . $e->getMessage()
            ], 500);
        }
    }

    public function dataTables()
    {
        $parcels = CoffeeParcelRegistry::all();

        return response()->json($parcels);
    }

    public function exportar(Request $request)
    {
        $year = $request->get('year', date('Y'));
        
        $parcelas = CoffeeParcelRegistry::where('version', $year)->get();
        
        $filename = "parcelas_{$year}_" . date('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($parcelas) {
            $file = fopen('php://output', 'w');
            
            // Headers del CSV
            fputcsv($file, [
                'ID', 'ID Kobo', 'Fecha Importación', 'Fecha Subida', 'Día',
                'Dispositivo', 'Usuario', 'Número Teléfono', 'ID Record',
                'Código Socio', 'Socio', 'Nombre Parcela', 'N° Parcela',
                'ID Parcela', 'Ciclo', 'Superficie', 'Cultivo Norte',
                'Propietario Norte', 'Cultivo Sur', 'Propietario Sur',
                'Cultivo Este', 'Propietario Este', 'Cultivo Oeste',
                'Propietario Oeste', 'Meta InstanceID', 'Meta RootUuid',
                'Meta InstanceName', 'Meta DeprecatedID', 'Versión Form',
                'Formhub UUID', 'XForm ID String', 'UUID', 'Status',
                'Submission Time', 'Submitted By', 'Versión'
            ]);
            
            // Datos
            foreach ($parcelas as $parcela) {
                fputcsv($file, [
                    $parcela->id,
                    $parcela->_id,
                    $parcela->created_at,
                    $parcela->starttime,
                    $parcela->today,
                    $parcela->deviceid,
                    $parcela->usuario,
                    $parcela->numero_telefono,
                    $parcela->id_record,
                    $parcela->codigo_socio,
                    $parcela->socio,
                    $parcela->nombre_parcela,
                    $parcela->n_parcela,
                    $parcela->id_parcela,
                    $parcela->ciclo,
                    $parcela->superficie,
                    $parcela->norte,
                    $parcela->norte_propietario,
                    $parcela->sur,
                    $parcela->sur_propietario,
                    $parcela->este,
                    $parcela->este_propietario,
                    $parcela->oeste,
                    $parcela->oeste_propietario,
                    $parcela->meta_instanceID,
                    $parcela->meta_rootUuid,
                    $parcela->meta_instanceName,
                    $parcela->meta_deprecatedID,
                    $parcela->__version__,
                    $parcela->formhub_uuid,
                    $parcela->_xform_id_string,
                    $parcela->_uuid,
                    $parcela->_status,
                    $parcela->_submission_time,
                    $parcela->_submitted_by,
                    $parcela->version
                ]);
            }
            
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    public function sincronizarAutomatico()
    {
        $config = [
            'year' => date('Y'),
            'username' => config('services.kobotoolbox.username'),
            'password' => config('services.kobotoolbox.password'),
            'asset_id' => config('services.kobotoolbox.asset_id'),
            'server_url' => config('services.kobotoolbox.server_url', 'https://eu.kobotoolbox.org')
        ];

        if (!$config['username'] || !$config['password'] || !$config['asset_id']){
            Log::error('Configuración de KoboToolbox incompleta para sincronización automática');
            return response()->json([
                'success' => false,
                'message' => 'Configuración incompleta'
            ]);
        }

        try{
            $importador = new ImportadorParcelas(
                $config['year'],
                $config['username'],
                $config['password'],
                $config['asset_id'],
                $config['server_url']
            );

            $resultado = $importador->importar();

            Log::info('Sincronización automática completada', $resultado);
            return response()->json($resultado);

        }catch (\Exception $e){
            Log::error("Error en sincronización automática: {$e->getMessage()}");
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }
}
