<?php

namespace App\Http\Controllers;

use App\Models\Producer;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Services\ImportadorProductores;
use Illuminate\Support\Facades\Validator;

use function Pest\version;
use function PHPUnit\Framework\isArray;
use Yajra\DataTables\Facades\DataTables;


class ProducerController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $currentYear = date('Y');
        
        $totalProducer = Producer::Where('version', $currentYear)->orWhere('version', $currentYear)->count();
        $totalActivos = Producer::where('activo', 1)->Wherein('version', [$currentYear])->count();
        $totalInactivos = Producer::where('activo', 0)->where('activo', '')->Wherein('version',[$currentYear])->count();
        $Organico = Producer::where('Estatus_actual', 'Organico')->where('activo', 1)->Wherein('version',[$currentYear])->count();
        $transicion1 = Producer::where('Estatus_actual', 'Transicion_1')->where('activo', 1)->Wherein('version',[$currentYear])->count();
        $transicion2 = Producer::where('Estatus_actual', 'Transicion_2')->where('activo', 1)->Wherein('version',[$currentYear])->count();
        $transicion3 = Producer::where('Estatus_actual', 'Transicion_3')->where('activo', 1)->Wherein('version',[$currentYear])->count();

        $producers = Producer::all();

        // if($producers->isEmpty()){
        //     $mensaje = [
        //         'tipo' => 'info',
        //         'texto' => 'No hay productores registrados. Importa Productores'
        //     ];
        // }

        return view('productores.index', compact('totalProducer','totalActivos','totalInactivos', 'Organico','transicion1','transicion2','transicion3','producers'));
    }
    /**
     * Importar productores desde KoBoToolbox
     */
    public function importarKobo(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'year' => 'required|integer|min:2020|max:' . (date('Y') + 1),
            'auth_method' => 'required|in:basic,token',
            'auth_credential' => 'required|string',
            'second_auth_credential' => 'required_if:auth_method,basic|nullable|string',
            'asset_id' => 'required|string',
            'server_url' => 'required|url'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Datos de entrada inválidos',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $validator->validated();
            
            // Crear instancia del importador
            $importador = new ImportadorProductores(
                year: $data['year'],
                assetId: $data['asset_id'],
                username: $data['auth_credential'],
                password: $data['second_auth_credential'] ?? null,
                serverUrl: $data['server_url']
            );

            // Ejecutar importación
            $resultado = $importador->importar();
            
            if ($resultado['success']) {
                Log::info('Importación exitosa desde KoBoToolbox', [
                    'year' => $data['year'],
                    'total_registros' => $resultado['total']
                ]);
            }

            return response()->json($resultado);

        } catch (\Exception $e) {
            Log::error('Error en importación KoBoToolbox', [
                'error' => $e->getMessage(),
                'year' => $request->year
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error interno del servidor: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener información del asset antes de importar
     */
    public function infoAsset(Request $request)
    {
       $validator = Validator::make($request->all(), [
            'auth_method' => 'required|in:basic,token',
            'auth_credential' => 'required|string',
            'second_auth_credential' => 'required_if:auth_method,basic|nullable|string',
            'asset_id' => 'required|string',
            'server_url' => 'required|url',
            'year' => 'required|integer|min:2020'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Datos de entrada inválidos',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $data = $validator->validated();
            
            $importador = new ImportadorProductores(
                year: $data['year'],
                assetId: $data['asset_id'],
                username: $data['auth_credential'],
                password: $data['second_auth_credential'] ?? null,
                serverUrl: $data['server_url']
            );

            $info = $importador->obtenerInfoAsset();
            return response()->json($info);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener información: ' . $e->getMessage()
            ], 500);
        }
    }


    // return view('productores.depuracion',['users' => $producers]);
    /**
     * Datos para DataTables
     */
    public function dataTables(Request $request)
    {
        $users = Producer::all();

        return response()->json($users);
        // $query = Producer::query();

       
    }
    /**
     * Obtener estadísticas de productores
     */
    public function estadisticas(Request $request)
    {
        $year = $request->get('year', date('Y'));
        // $currentYear = date('Y');
        
        try {
            $baseQuery = Producer::where('version', $year);
            
            $estadisticas = [
                'total' => Producer::whereIn('version', [$year])->count(),
                
                'activos' => Producer::where('activo', 1)
                    ->whereIn('version', [$year])
                    ->count(),
                
                'inactivos' => Producer::where(function($q) {
                        $q->where('activo', 0)->orWhere('activo', '');
                    })
                    ->whereIn('version', [$year])
                    ->count(),
                
                'por_estatus' => [
                    'Organico' => Producer::where('estatus', 'Organico')
                        ->where('activo', 1)
                        ->whereIn('version', [$year])
                        ->count(),
                    'Natural' => Producer::where('estatus', 'Natural')
                        ->where('activo', 1)
                        ->whereIn('version', [$year])
                        ->count(),
                    'Transicion_1' => Producer::where('estatus', 'Transicion_1')
                        ->where('activo', 1)
                        ->whereIn('version', [$year])
                        ->count(),
                    'Transicion_2' => Producer::where('estatus', 'Transicion_2')
                        ->where('activo', 1)
                        ->whereIn('version', [$year])
                        ->count(),
                    'Transicion_3' => Producer::where('estatus', 'Transicion_3')
                        ->where('activo', 1)
                        ->whereIn('version', [$year])
                        ->count(),
                ],
                
                'por_region' => [
                    'Chenalho' => Producer::whereIn('municipio', ['Chenalho', 'Chenalhó'])
                        ->where('activo', 1)
                        ->whereIn('version', [$year])
                        ->count(),
                    'San_Juan_Cancuc' => Producer::whereIn('municipio', ['San_Juan_Cancuc', 'San Juan Cancuc'])
                        ->where('activo', 1)
                        ->whereIn('version', [$year])
                        ->count(),
                    'Tenejapa' => Producer::where('municipio', 'Tenejapa')
                        ->where('activo', 1)
                        ->whereIn('version', [$year])
                        ->count(),
                    'Pantelho' => Producer::where('municipio', 'Pantelho')
                        ->where('activo', 1)
                        ->whereIn('version', [$year])
                        ->count(),
                    'Oxchuc' => Producer::where('municipio', 'Oxchuc')
                        ->where('activo', 1)
                        ->whereIn('version', [$year])
                        ->count(),
                ]


                    // ->whereNotNull('municipio')
                    // ->where('municipio', '!=', '')
                    // ->where('activo',1)
                    // ->groupBy('municipio')
                    // ->selectRaw('municipio, count(*) as total')
                    // ->orderBy('total', 'desc')
                    // ->pluck('total', 'municipio')
                    
                    // ->toArray()
            ];

            return response()->json($estadisticas);

        } catch (\Exception $e) {
            Log::error('Error al obtener estadísticas', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error al obtener estadísticas'
            ], 500);
        }
    }
    /**
     * Exportar productores
     */
    public function exportar(Request $request)
    {
        $year = $request->get('year', date('Y'));
        
        $productores = Producer::where('version', $year)->get();
        
        $filename = "productores_{$year}_" . date('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($productores) {
            $file = fopen('php://output', 'w');
            
            // Headers del CSV
            fputcsv($file, [
                'ID', 'ID Productor', 'Localidad', 'Municipio', 'Región', 
                'Socio', 'Grupo Trabajo', 'CURP', 'RFC', 'INE',
                'Folio Cafetalero', 'Folio SURI', 'Fecha Nacimiento', 
                'Género', 'Estatus', 'Activo', 'Número Parcelas',
                'Fecha Importación', 'Fecha Envío'
            ]);
            
            // Datos
            foreach ($productores as $productor) {
                fputcsv($file, [
                    $productor->id,
                    $productor->id_productor,
                    $productor->localidad,
                    $productor->municipio,
                    $productor->region,
                    $productor->socio,
                    $productor->grupo_trabajo,
                    $productor->curp,
                    $productor->rfc,
                    $productor->ine,
                    $productor->folio_cafetalero,
                    $productor->folio_suri,
                    $productor->fecha_nacimiento,
                    $productor->genero,
                    $productor->estatus,
                    $productor->activo ? 'Sí' : 'No',
                    $productor->numero_parcelas,
                    $productor->fecha_importacion,
                    $productor->_submission_time
                ]);
            }
            
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
     /**
     * Sincronización programada (para usar con cron)
     */
    public function sincronizarAutomatico()
    {
        $config = [
            'year' => date('Y'),
            'username' => config('services.kobotoolbox.username'),
            'password' => config('services.kobotoolbox.password'),
            'asset_id' => config('services.kobotoolbox.asset_id'),
            'server_url' => config('services.kobotoolbox.server_url', 'https://eu.kobotoolbox.org')
        ];

        if (!$config['username'] || !$config['password'] || !$config['asset_id']) {
            Log::error('Configuración de KoBoToolbox incompleta para sincronización automática');
            return response()->json([
                'success' => false,
                'message' => 'Configuración incompleta'
            ]);
        }

        try {
            $importador = new ImportadorProductores(
                $config['year'],
                $config['username'],
                $config['password'],
                $config['asset_id'],
                $config['server_url']
            );

            $resultado = $importador->importar();
            
            Log::info('Sincronización automática completada', $resultado);
            
            return response()->json($resultado);

        } catch (\Exception $e) {
            Log::error("Error en sincronización automática: {$e->getMessage()}");
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        return $this->index();
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, int $anio)
    {
        Producer::where('anio', $anio)->delete();
        $data = Http::get('response')->json();

        $this->update($data, $anio);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request)
    {
        $version = $request->input('version');

        if(!$version){
            return redirect()->back()->with('error', 'Indica que versión eliminar');
        }

        $eliminados = Producer::where('version', $version)->delete();

        return redirect()->back()->with('success', "Se eliminaron {$eliminados} registros des la verion {$version}");

    }
  
}

