<?php

namespace App\Services\Importers;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

abstract class KoboImporterBase
{
    protected string $koboUrl;
    protected string $koboToken;
    protected string $formId;

    public function __construct()
    {
        $this->koboUrl = config('services.kobo.url');
        $this->koboToken = config('services.kobo.token');
    }

    /**
     * Obtener datos de KoboToolbox
     */
    protected function fetchKoboData(): array
    {
        $response = Http::withToken($this->koboToken)
            ->get("{$this->koboUrl}/api/v2/assets/{$this->formId}/data/");

        if ($response->failed()) {
            Log::error('Error fetching Kobo data', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            throw new \Exception('Failed to fetch data from KoboToolbox');
        }

        return $response->json()['results'] ?? [];
    }

    /**
     * Transformar datos de Kobo al formato deseado
     */
    abstract protected function transformData(array $koboData): array;

    /**
     * Guardar datos en MongoDB
     */
    abstract protected function saveToMongoDB(array $transformedData): void;

    /**
     * Ejecutar importación completa
     */
    public function import(): array
    {
        try {
            $koboData = $this->fetchKoboData();
            
            $stats = [
                'total' => count($koboData),
                'imported' => 0,
                'failed' => 0,
                'errors' => []
            ];

            foreach ($koboData as $record) {
                try {
                    $transformed = $this->transformData($record);
                    $this->saveToMongoDB($transformed);
                    $stats['imported']++;
                } catch (\Exception $e) {
                    $stats['failed']++;
                    $stats['errors'][] = [
                        'record' => $record['_id'] ?? 'unknown',
                        'error' => $e->getMessage()
                    ];
                    Log::error('Error importing record', [
                        'record' => $record,
                        'error' => $e->getMessage()
                    ]);
                }
            }

            return $stats;
        } catch (\Exception $e) {
            Log::error('Import failed', ['error' => $e->getMessage()]);
            throw $e;
        }
    }
}