<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use App\Models\Georeferencing;

use function Pest\Laravel\json;

class ImportadorGeoreferenciaciones extends ImportadorBase
{
    protected string $modelClass = Georeferencing::class;
    protected string $idExterno = 'georeferenciacion/codigo_parcela';

    protected array $mapeoCampos = [
        '_id' => '_id',
        'start' => 'start',
        'end' => 'end',
        'today' => 'today',
        'username' => 'username',
        'deviceid' => 'deviceid',
        'phonenumber' => 'phonenumber',
        'hora' => 'hora',
        'horas' => 'horas',
        'minutos' => 'minutos',
        'segundos' => 'segundos',
        'anos' => 'anos',
        'mes' => 'mes',
        'dias' => 'dias',
        'id_record' => 'id_record',
        'promotor' => 'bienvenida/promotor',
        'dni' => 'georeferenciacion/dni',
        'socio' => 'georeferenciacion/socio',
        'localidad' => 'georeferenciacion/localidad',
        'municipio' => 'georeferenciacion/municipio',
        'numero_parcela' => 'georeferenciacion/numero_parcela',
        'codigo_parcela' => 'georeferenciacion/codigo_parcela',
        'nombre_parcela' => 'georeferenciacion/nombre_parcela',
        'anio_incio_cultivo' => 'georeferenciacion/anio_incio_cultivo',
        'gps' => 'georeferenciacion/gps',
        'poligono' => 'georeferenciacion/poligono',
        'area_gps' => 'georeferenciacion/area_gps',
        'area_gps_ha' => 'georeferenciacion/area_gps_ha',
        'area_gps_display' => 'georeferenciacion/area_gps_display',
        'meta_instanceID' => 'meta/instanceID',
        'meta_rootUuid' => 'meta/rootUuid',
        'meta_instanceName' => 'meta/instanceName',
        'meta_deprecatedID' => 'meta/deprecatedID',
        'starttime' => 'starttime',
        'endtime' => 'endtime',
        'p_nombre' => 'p_nombre',
        'ciclo_codigo' => 'ciclo_codigo',
        'subscriberid' => 'subscriberid',
        'simid' => 'simid',
        'formhub_uuid' => 'formhub/uuid',
        '__version__' => '__version__',
        '_xform_id_string' => '_xform_id_string',
        '_uuid' => '_uuid',
        '_attachments' => '_attachments',
        '_status' => '_status',
        '_geolocation' => '_geolocation',
        '_submission_time' => '_submission_time',
        '_tags' => '_tags',
        '_notes' => '_notes',
        '_validation_status' => '_validation_status',
        '_submitted_by' => '_submitted_by'
    ];
    public function __construct(
        int $year,
        string $assetId,
        ?string $username = null,
        ?string $password = null,
        string $serverUrl = 'https://eu.kobotoolbox.org'
    )

    {
        parent::__construct($year);

        $this->urlApi = "{$serverUrl}/api/v2/assets/{$assetId}/data/?format=json";

        $this->setCredenciales(
            $username ?: $this->username,
            $password ?: $this->password
        );

        $this->setHeaders([
            'Accept' => 'application/json',
            'Content-Type' => 'application/json'
        ]);
    }
    protected function extraerResultados(array $datos): array
    {
        return $datos['results'] ?? [];
    }
    protected function mapearDatos(array $datoApi): array
    {
        $datosMapeados = [];

        foreach ($this->mapeoCampos as $campoLocal => $campoApi) {
            $valor = $this->obtenerValorAnidado($datoApi, $campoApi);

            // Ver arrays problemáticos (debugging)
            if (is_array($valor) || is_object($valor)) {
                Log::info("Campo array/objeto detectado: {$campoLocal}", ['valor' => $valor]);
            }

            // Normalizamos todos los valores antes de asignar
            $datosMapeados[$campoLocal] = $this->normalizarValor($valor);
        }
        
    

        // Procesamiento especial para fechas
        $datosMapeados['start'] = $this->parsearFechaHora($datosMapeados['start']);
        $datosMapeados['end'] = $this->parsearFechaHora($datosMapeados['end']);
        $datosMapeados['today'] = $this->parsearFecha($datosMapeados['today']);
        $datosMapeados['starttime'] = $this->parsearFechaHora($datosMapeados['starttime']);
        $datosMapeados['endtime'] = $this->parsearFechaHora($datosMapeados['endtime']);
        $datosMapeados['_submission_time'] = $this->parsearFechaHora($datosMapeados['_submission_time']);
        $datosMapeados['anio_incio_cultivo'] = $this->parsearFecha($datosMapeados['anio_incio_cultivo']);

        return $datosMapeados;
    }


    protected function obtenerValorAnidado(array $datos, string $campo)
    {
        // Caso 1: la clave existe literal en el array
        if (array_key_exists($campo, $datos)) {
            return $datos[$campo];
        }
        
        $separadores = ['/', '.'];

        foreach ($separadores as $separador) {
            if (strpos($campo, $separador) !== false) {
                $claves = explode($separador, $campo);
                $valor = $datos;

                foreach ($claves as $clave) {
                    if (is_array($valor) && array_key_exists($clave, $valor)) {
                        $valor = $valor[$clave];
                    } else {
                        return null;
                    }
                }
                return $valor;
            }
        }
        // Si no hay separadores, es un campo simple
        return $datos[$campo] ?? null;
    }

    protected function parsearFecha(?string $fecha): ?string
    {
        if (!$fecha) return null;

        try {
            $timestamp = strtotime($fecha);
            return $timestamp ? date('Y-m-d', $timestamp) : null;
        } catch (\Exception $e) {
            Log::warning("Error al parsear fecha: {$fecha}", ['exception' => $e->getMessage()]);
            return null;
        }
    }

    protected function parsearFechaHora(?string $fechaHora): ?string
    {
        if (!$fechaHora) return null;

        try {
            $timestamp = strtotime($fechaHora);
            return $timestamp ? date('Y-m-d H:i:s', $timestamp) : null;
        } catch (\Exception $e) {
            Log::warning("Error al parsear fecha-hora: {$fechaHora}", ['exception' => $e->getMessage()]);
            return null;
        }
    }

    private function extraerAssetIdDeUrl(): ?string
    {
        if (preg_match('/assets\/([^\/]+)\/data/', $this->urlApi, $matches)){
            return $matches[1];
        }
        return null;
    }

    public function obtenerInfoAsset(): array
    {
        try {
            $assetUrl = str_replace('/data/?format=json', '', $this->urlApi);
            
            $httpClient = Http::timeout(30);

            if ($this->username && $this->password) {
                $httpClient = $httpClient->withBasicAuth($this->username, $this->password);
            }
            
            if (!empty($this->headers)) {
                $httpClient = $httpClient->withHeaders($this->headers);
            }
            
            $response = $httpClient->get($assetUrl);
            
            if (!$response->successful()) {
                return [
                    'success' => false, 
                    'message' => $this->getErrorMessage($response->status())
                ];
            }
            
            $data = $response->json();
            return [
                'success' => true,
                'name' => $data['name'] ?? 'Sin nombre',
                'deployment_count' => $data['deployment__submission_count'] ?? 0,
                'date_modified' => $data['date_modified'] ?? null,
                'asset_type' => $data['asset_type'] ?? 'survey'
            ];
            
        } catch (\Exception $e) {
            Log::error("Error al obtener info del asset: {$e->getMessage()}");
            return [
                'success' => false, 
                'message' => 'Error al obtener información: ' . $e->getMessage()
            ];
        }
    }

    //FUNCION PARA SOBRE ESCRIBIR Y PASAR LA VALIDACIÓN SI SOCIO EXISTE
    protected function procesarRegistro(array $datoApi)
    {
        $datosMapeados = $this->mapearDatos($datoApi);
        $datosMapeados['version'] = $this->year;

        $valorIdExterno = $this->obtenerValorAnidado($datoApi, 'georeferenciacion/codigo_parcela');

        if (!$valorIdExterno) {
            Log::error("No se pudo obtener el ID externo para el registro", ['dato' => $datoApi]);
            return;
        }

        // Si no viene codigo_parcela, generar uno ficticio
        if (empty($valorIdExterno)) {
            $valorIdExterno = 'SIN_CODIGO_' . ($datoApi['_id'] ?? uniqid());
            $datosMapeados['codigo_parcela'] = $valorIdExterno;

            Log::warning("Registro sin codigo_parcela, asignado: {$valorIdExterno}", [
                '_id' => $datoApi['_id'] ?? null
            ]);
        }

        $model = $this->modelClass::where('codigo_parcela', $valorIdExterno)
            ->where('version', $this->year)
            ->first();

        if ($model) {
            $model->update($datosMapeados);
            Log::info(" Actualizado registro con ID: {$valorIdExterno}");
        } else {
            try {
                Log::info('Insertando registro', ['_id' => $datoApi['_id'] ?? null]);
                $this->modelClass::create($datosMapeados);
                Log::info("Creado nuevo registro con ID: {$valorIdExterno}");
            } catch (\Exception $e) {
                Log::error('Error al insertar registro', [
                    '_id' => $datoApi['_id'] ?? null,
                    'codigo_parcela' => $valorIdExterno,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
            }
        }
    } 
            
    protected function validarParcela($codigoSocio): bool 
    {
        
        if (empty($codigoSocio)){
            return false; 
            log::warning("Parcela no encontrada");
        }
        return \App\Models\Producer::where('id_productor', $codigoSocio)->exists();

    }
    protected function obtenerValorIdExterno(array $datoApi)
{
    // La clave anidada específica para georreferenciación
    $valor = $this->obtenerValorAnidado($datoApi, 'georeferenciacion/codigo_parcela');

    // Si no existe, genera uno temporal
    if (empty($valor)) {
        $valor = 'SIN_CODIGO_' . ($datoApi['_id'] ?? uniqid());
    }

    return $valor;
}
    

    public function validarCredenciales(): bool
    {
        $info = $this->obtenerInfoAsset();
        return $info['success'] ?? false;
    }


    protected function normalizarValor($valor)
    {
        if ($valor === null) {
            return null;
        }

        // Si es array, convertir a JSON
        if (is_array($valor)) {
            return json_encode($valor, JSON_UNESCAPED_UNICODE);
        }

        // Si es Collection, convertir a JSON
        if ($valor instanceof \Illuminate\Support\Collection) {
            return json_encode($valor->all(), JSON_UNESCAPED_UNICODE);
        }

        // Si es objeto, intentar convertir a string o JSON
        if (is_object($valor)) {
            if (method_exists($valor, '__toString')) {
                return (string) $valor;
            }
            return json_encode($valor, JSON_UNESCAPED_UNICODE);
        }

        // Si es booleano, convertir a string
        if (is_bool($valor)) {
            return $valor ? '1' : '0';
        }

        // Para escalares (string, int, float), devolver tal como están
        return $valor;
    }
    
}