<?php

namespace App\Services\Importers;

use App\Models\Inspeccion;
use Illuminate\Support\Facades\DB;

class InspeccionKoboImporter extends KoboImporterBase
{
    protected string $formId;

    public function __construct()
    {
        parent::__construct();
        $this->formId = config('services.kobo.form_ids.inspeccion');
    }

    protected function transformData(array $koboData): array
    {
        return [
            'id_inspeccion' => $koboData['id_inspeccion'] ?? null,
            'dni' => $koboData['productor/dni'] ?? null,
            'fecha_visita' => $koboData['info_general/fecha_visita'] ?? null,
            'hora_inicio' => $koboData['info_general/hora_inicio'] ?? null,
            'inspector' => $koboData['info_general/inspector'] ?? null,
            
            // GPS como array
            'gps_productor' => $this->parseGPS($koboData['info_general/gps_productor'] ?? null),
            
            // Información general agrupada
            'info_general' => [
                'productor_nuevo' => $koboData['info_general/productor_nuevo'] ?? null,
                'otra_organizacion' => $koboData['info_general/otra_organizacion'] ?? null,
                'nombre_otra_organizacion' => $koboData['info_general/nombre_otra_organizacion'] ?? null,
            ],
            
            // Datos del productor
            'productor' => [
                'socio' => $koboData['productor/socio'] ?? null,
                'curp' => $koboData['productor/curp'] ?? null,
                'rfc' => $koboData['productor/rfc'] ?? null,
                'ine' => $koboData['productor/ine'] ?? null,
                'municipio' => $koboData['productor/municipio'] ?? null,
                'localidad' => $koboData['productor/localidad'] ?? null,
                'telefono' => $koboData['productor/telefono'] ?? null,
                'genero' => $koboData['productor/genero'] ?? null,
                'baja_productor' => $koboData['productor/baja_productor'] ?? null,
                'tenencia_tierra' => $koboData['productor/tenencia_tierra'] ?? null,
            ],
            
            // Áreas generales
            'area_general' => [
                'tiene_area_autoconsumo' => $koboData['area_general/tiene_area_autoconsumo'] ?? null,
                'area_autoconsumo' => $koboData['area_general/area_autoconsumo'] ?? null,
                'tiene_area_potrero' => $koboData['area_general/tiene_area_potrero'] ?? null,
                'area_potrero' => $koboData['area_general/area_potrero'] ?? null,
                'tiene_area_bosque' => $koboData['area_general/tiene_area_bosque'] ?? null,
                'area_bosque' => $koboData['area_general/area_bosque'] ?? null,
            ],
            
            // Parcelas (array de objetos)
            'parcelas' => $this->extractParcelas($koboData),
            
            // Resumen
            'resumen_parcelas' => [
                'parcelas_totales' => $koboData['resumen_parcelas/parcelas_totales'] ?? null,
                'area_total' => $koboData['resumen_parcelas/area_total'] ?? null,
                'plantas_totales' => $koboData['resumen_parcelas/plantas_totales'] ?? null,
                'kg_estimados_totales' => $koboData['resumen_parcelas/kg_estimados_totales'] ?? null,
            ],
            
            // Metadata
            'metadata' => [
                'kobo_id' => $koboData['_id'] ?? null,
                'submission_time' => $koboData['_submission_time'] ?? null,
                'submitted_by' => $koboData['_submitted_by'] ?? null,
            ]
        ];
    }

    protected function parseGPS(?string $gpsString): ?array
    {
        if (!$gpsString) {
            return null;
        }

        $parts = explode(' ', $gpsString);
        
        return [
            'latitude' => (float)($parts[0] ?? 0),
            'longitude' => (float)($parts[1] ?? 0),
            'altitude' => (float)($parts[2] ?? 0),
            'accuracy' => (float)($parts[3] ?? 0),
        ];
    }

    protected function extractParcelas(array $koboData): array
    {
        $parcelas = [];
        
        // Extraer hasta 9 parcelas
        for ($i = 1; $i <= 9; $i++) {
            $prefix = "parcela_{$i}";
            
            if (isset($koboData["{$prefix}/p{$i}_nombre"])) {
                $parcelas[] = [
                    'numero' => $i,
                    'codigo' => $koboData["{$prefix}/p{$i}_codigo"] ?? null,
                    'nombre' => $koboData["{$prefix}/p{$i}_nombre"] ?? null,
                    'area' => $koboData["{$prefix}/p{$i}_area"] ?? null,
                    'activa' => $koboData["{$prefix}/p{$i}_activa"] ?? null,
                    'variedades' => $koboData["{$prefix}/p{$i}_variedades"] ?? null,
                    'sombra' => $koboData["{$prefix}/p{$i}_tipo_sombra"] ?? null,
                    'plantas_totales' => $koboData["{$prefix}/p{$i}_n_plantas_tot"] ?? null,
                    'kg_estimado' => $koboData["{$prefix}/p{$i}_kg_pergamino_tot"] ?? null,
                    'riesgo' => $koboData["{$prefix}/p{$i}_riesgo"] ?? null,
                    'observaciones' => $koboData["{$prefix}/p{$i}_observaciones"] ?? null,
                ];
            }
        }
        
        return $parcelas;
    }

    protected function saveToMongoDB(array $transformedData): void
    {
        // Usar updateOrCreate para evitar duplicados
        Inspeccion::updateOrCreate(
            ['id_inspeccion' => $transformedData['id_inspeccion']],
            $transformedData
        );
    }
}