<?php

// app/Services/ImportadorParcelas.php

namespace App\Services;

use App\Models\CoffeeParcelRegistry;
use Illuminate\Support\Facades\Log;
use App\Models\Parcela;
use Illuminate\Support\Facades\Http;
use Illuminate\Validation\Rules\In;
use phpDocumentor\Reflection\Types\Parent_;

use function Pest\Laravel\json;

class ImportadorParcelas extends ImportadorBase
{
    protected string $modelClass = CoffeeParcelRegistry::class;
    protected string $idExterno = '_id';

    protected array $mapeoCampos = [
        '_id' => '_id',
        'starttime' => 'starttime',
        'endtime' => 'endtime',
        'today' => 'today',
        'deviceid' => 'deviceid',
        'usuario' => 'usuario',
        'numero_telefono' => 'numero_telefono',
        'id_record' => 'id_record',
        'codigo_socio' => 'codigo_socio',
        'socio' => 'socio',
        'nombre_parcela' => 'nombre_parcela',
        'n_parcela' => 'n_parcela',
        'id_parcela' => 'id_parcela',
        'ciclo' => 'ciclo',
        'superficie' => 'superficie',
        'norte' => 'norte',
        'norte_propietario' => 'norte_propietario',
        'sur' => 'sur',
        'sur_propietario' => 'sur_propietario',
        'este' => 'este',
        'este_propietario' => 'este_propietario',
        'oeste' => 'oeste',
        'oeste_propietario' => 'oeste_propietario',
        'meta_instanceID' => 'meta/instanceID',
        'meta_rootUuid' => 'meta/rootUuid',
        'meta_instanceName' => 'meta/instanceName',
        'meta_deprecatedID' => 'meta/deprecatedID',
        '__version__' => '__version__',
        'formhub_uuid' => 'formhub/uuid',
        '_xform_id_string' => '_xform_id_string',
        '_uuid' => '_uuid',
        '_status' => '_status',
        '_submission_time' => '_submission_time',
        '_submitted_by' => '_submitted_by',
        'version' => 'version',
    ];

    public function __construct(
        int $year,
        string $assetId,
        ?string $username = null,
        ?string $password = null,
        string $serverUrl = 'https://eu.kobotoolbox.org'
    )
    {
        parent::__construct($year);

        //Contruir la URL de la API

        $this->urlApi = "{$serverUrl}/api/v2/assets/{$assetId}/data/?format=json";

        $this->setCredenciales(
            $username ?: $this->username,
            $password ?: $this->password
        );

        $this->setHeaders([
            'Accept' => 'application/json',
            'Content-Type' => 'application/json'
        ]);
    }

    protected function extraerResultados(array $datos): array
    {
        return $datos['results'] ?? [];
    }

    protected function mapearDatos(array $datoApi): array
    {
        $datosMapeados = [];

        foreach ($this->mapeoCampos as $campoLocal => $campoApi){
            $valor = $this->obtenerValorAnidado($datoApi, $campoApi);
            $datosMapeados[$campoLocal] = $valor;
        }
        
        // foreach ($this->mapeoCampos as $campoLocal => $campoApi) {
        //     if (array_key_exists($campoApi, $datoApi)) {
        //         $datosMapeados[$campoLocal] = $datoApi[$campoApi];
        //     }
        // }

        // Procesamiento especial para fechas y números
        $datosMapeados['starttime'] = $this->parsearFechaHora($datoApi['starttime'] ?? null);
        $datosMapeados['endtime'] = $this->parsearFechaHora($datoApi['endtime'] ?? null);
        $datosMapeados['today'] = $this->parsearFecha($datoApi['today'] ?? null);
        $datosMapeados['_submission_time'] = $this->parsearFechaHora($datoApi['_submission_time'] ?? null);
        $datosMapeados['superficie'] = isset($datoApi['superficie']) ? (float)$datoApi['superficie'] : 0;

        //Agregar campos especificos de KoBoToolbox
        $datosMapeados['kobo_asset_id'] = $this->extraerAssetIdDeUrl();
        $datosMapeados['fecha_ultima_importacion'] = now();
        $datosMapeados['datos_raw'] = json_encode($datoApi); // Backup de datos originales

        return $datosMapeados;
    }

    protected function obtenerValorAnidado(array $datos, string $campo)
    {
        //Caso 1: la clave existe literal en el array (meta/instanceID)
        if(array_key_exists($campo, $datos)){
            return $datos[$campo];
        }
        
        $separadores = ['/', '.'];

        foreach ($separadores as $separdor){
            if(strpos($campo, $separdor) !== false){
                $claves = explode($separdor, $campo);
                $valor = $datos;

                foreach ($claves as $clave){
                    if (is_array($valor) && array_key_exists($clave, $valor)){
                        $valor = $valor[$clave];
                    }else{
                        return null;
                    }
                }
                return $valor;
            }
        }
        // Si no hay separadores, es un campo simple
        return $datos[$campo] ?? null;
    }

    protected function parsearFecha(?string $fecha): ?string
    {
        if (!$fecha) return null;
        
        try {
            // Eliminar la parte de tiempo si existe
            $fecha = explode('T', $fecha)[0];
            
            // Validar formato de fecha
            $date = \DateTime::createFromFormat('Y-m-d', $fecha);
            if ($date && $date->format('Y-m-d') === $fecha) {
                return $fecha;
            }
            
            // Si no es formato Y-m-d, intentar parsearlo
            $timestamp = strtotime($fecha);
            return $timestamp ? date('Y-m-d', $timestamp) : null;
        } catch (\Exception $e) {
            Log::warning("Error al parsear fecha: {$fecha}", ['exception' => $e->getMessage()]);
            return null;
        }
    }

    protected function parsearFechaHora(?string $fechaHora): ?string
    {
        if (!$fechaHora) return null;
        
        try {
            // Convertir fecha ISO 8601 a formato MySQL datetime
            $date = new \DateTime($fechaHora);
            return $date->format('Y-m-d H:i:s');
        } catch (\Exception $e) {
            Log::warning("Error al parsear fecha-hora: {$fechaHora}", ['exception' => $e->getMessage()]);
            return null;
        }
    }

    private function extraerAssetIdDeUrl(): ?string
    {
        if (preg_match('/assets\/([^\/]+)\/data/', $this->urlApi, $matches)){
            return $matches[1];
        }
        return null;
    }

    public function obtenerInfoAsset(): array
    {
        try {
            $assetUrl = str_replace('/data/?format=json', '', $this->urlApi);
            
            $httpClient = Http::timeout(30);

            // Usar las mismas credenciales configuradas
            if ($this->username && $this->password) {
                $httpClient = $httpClient->withBasicAuth($this->username, $this->password);
            }
            
            if (!empty($this->headers)) {
                $httpClient = $httpClient->withHeaders($this->headers);
            }
            
            $response = $httpClient->get($assetUrl);
            
            if (!$response->successful()) {
                return [
                    'success' => false, 
                    'message' => $this->getErrorMessage($response->status())
                ];
            }
            
            
            $data = $response->json();
            return [
                'success' => true,
                'name' => $data['name'] ?? 'Sin nombre',
                'deployment_count' => $data['deployment__submission_count'] ?? 0,
                'date_modified' => $data['date_modified'] ?? null,
                'asset_type' => $data['asset_type'] ?? 'survey'
            ];
            
        } catch (\Exception $e) {
            Log::error("Error al obtener info del asset: {$e->getMessage()}");
            return [
                'success' => false, 
                'message' => 'Error al obtener información: ' . $e->getMessage()
            ];
        }
    }

    //FUNCION PARA SOBRE ESCRIBIR Y PASAR LA VALIDACIÓN SI SOCIO EXISTE
    protected function procesarRegistro(array $datoApi)
     { 
        $datosMapeados = $this->mapearDatos($datoApi);
        
        //Validar que el productor existe 
        if(!$this->validarProductorExiste($datosMapeados['codigo_socio'])){ 
            Log::warning("Productor no encontrado - Saltando registro",[ 'codigo_socio' => $datosMapeados['codigo_socio'] ?? null ]); 
            return; 
        } 
            parent::procesarRegistro($datoApi); 
    } 
            
    protected function validarProductorExiste($codigoSocio): bool 
    { 
        if(empty($codigoSocio)){ return false; } 
        return \App\Models\Producer::where('id_productor', $codigoSocio)->exists(); 
    }

    public function validarCredenciales(): bool
    {
        $info = $this->obtenerInfoAsset();
        return $info['success'] ?? false;
    }
}