<?php
// app/Services/ImportadorProductores.php
// VERSIÓN CORREGIDA - Reemplaza todo el archivo

namespace App\Services;

use App\Models\Producer;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class ImportadorProductores extends ImportadorBase
{
    protected string $modelClass = Producer::class;
    protected string $idExterno = '_id';

    protected array $mapeoCampos = [
        '_id' => '_id',
        'id_productor' => 'id_productor',
        'fecha_importacion' => 'fecha_importacion',
        'localidad' => 'localidad',
        'municipio' => 'municipio',
        'region' => 'region',
        'socio' => 'socio',
        'grupo_trabajo' => 'grupo_trabajo',
        'curp' => 'curp',
        'rfc' => 'rfc',
        'ine' => 'ine',
        'folio_cafetalero' => 'folio_cafetalero',
        'folio_suri' => 'folio_suri',
        'fecha_nacimiento' => 'fecha_nacimiento',
        'genero' => 'genero',
        'estatus' => 'estatus',
        'categoria_pasada' => 'categoria_pasada',
        'ingreso' => 'ingreso',
        'transicion' => 'transicion',
        'numero_parcelas' => 'numero_parcelas',
        'activo' => 'activo',
        'meta_instanceID' => 'meta/instanceID',
        'meta_rootUuid' => 'meta/rootUuid',
        'meta_instanceName' => 'meta/instanceName',
        'meta_deprecatedID' => 'meta/deprecatedID',
        '__version__' => '__version__',
        'formhub_uuid' => 'formhub/uuid',
        '_xform_id_string' => '_xform_id_string',
        '_uuid' => '_uuid',
        '_status' => '_status',
        '_submission_time' => '_submission_time',
        '_submitted_by' => '_submitted_by'
    ];

    public function __construct(
        int $year, 
        string $assetId,
        ?string $username = null, 
        ?string $password = null, 
        string $serverUrl = 'https://eu.kobotoolbox.org')
    {
        parent::__construct($year);

        // Construir la URL de la API
        $this->urlApi = "{$serverUrl}/api/v2/assets/{$assetId}/data/?format=json";
        
        $this->setCredenciales(
            $username ?: $this->username,
            $password ?: $this->password
        );
        
        $this->setHeaders([
            'Accept' => 'application/json',
            'Content-Type' => 'application/json'
        ]);
    }

    /**
     * KoBoToolbox v2 API devuelve los datos en 'results'
     */
    protected function extraerResultados(array $datos): array
    {
        return $datos['results'] ?? [];
    }

    protected function mapearDatos(array $datoApi): array
    {
        $datosMapeados = [];
        
        // Mapear campos básicos
        foreach ($this->mapeoCampos as $campoLocal => $campoApi) {
            $valor = $this->obtenerValorAnidado($datoApi, $campoApi);
            $datosMapeados[$campoLocal] = $valor;
        }

        $datosMapeados['Estatus_actual'] = $datoApi['Estatus_actual_2025'] 
                                        ?? $datoApi['categoria25']
                                        ?? $datoApi['Estatus_actual']
                                        ?? $datoApi['categoria_actual']
                                        ?? null;

        // Procesamiento especial para fechas (SOLO FECHA, SIN HORA)
        $datosMapeados['fecha_importacion'] = $this->parsearFecha($datoApi['fecha_importacion'] ?? null);
        $datosMapeados['fecha_nacimiento'] = $this->parsearFecha($datoApi['fecha_nacimiento'] ?? null);
        $datosMapeados['ingreso'] = $this->parsearFecha($datoApi['ingreso'] ?? null);
        $datosMapeados['transicion'] = $this->parsearFecha($datoApi['transicion'] ?? null);
        
        // Este SÍ incluye hora porque es _submission_time
        $datosMapeados['_submission_time'] = $this->parsearFechaHora($datoApi['_submission_time'] ?? null);
        
        // Convertir activo a booleano
        $datosMapeados['activo'] = isset($datoApi['activo']) ?
            (bool) filter_var($datoApi['activo'], FILTER_VALIDATE_BOOLEAN) : false;

        // Agregar campos específicos de KoBoToolbox
        $datosMapeados['kobo_asset_id'] = $this->extraerAssetIdDeUrl();
        
        // ⚠️ CORRECCIÓN IMPORTANTE: usar format('Y-m-d') para solo guardar fecha
        $datosMapeados['fecha_ultima_importacion'] = now()->format('Y-m-d');
        
        $datosMapeados['datos_raw'] = json_encode($datoApi);

        return $datosMapeados;
    }

    /**
     * Obtener valor de campo anidado usando notación de punto o slash
     */
    protected function obtenerValorAnidado(array $datos, string $campo)
    {
        // Caso 1: la clave existe literal en el array (meta/instanceID)
        if(array_key_exists($campo, $datos)){
            return $datos[$campo];
        }
        
        // Caso 2: manejar notación de punto o slash para arrays anidados
        $separadores = ['/', '.'];

        foreach ($separadores as $separador) {
            if (strpos($campo, $separador) !== false) {
                $claves = explode($separador, $campo);
                $valor = $datos;

                foreach ($claves as $clave) {
                    if (is_array($valor) && array_key_exists($clave, $valor)) {
                        $valor = $valor[$clave];
                    } else {
                        return null;
                    }
                }
                return $valor;
            }
        }

        // Caso 3: clave simple
        return $datos[$campo] ?? null;
    }

    /**
     * Parsear fecha - devuelve SOLO fecha en formato Y-m-d
     */
    protected function parsearFecha(?string $fecha): ?string
    {
        if (!$fecha) return null;
        
        try {
            // Eliminar la parte de tiempo si existe (todo después de 'T')
            $fecha = explode('T', $fecha)[0];
            
            // Validar formato de fecha Y-m-d
            $date = \DateTime::createFromFormat('Y-m-d', $fecha);
            if ($date && $date->format('Y-m-d') === $fecha) {
                return $fecha;
            }
            
            // Si no es formato Y-m-d, intentar parsearlo
            $timestamp = strtotime($fecha);
            return $timestamp ? date('Y-m-d', $timestamp) : null;
        } catch (\Exception $e) {
            Log::warning("Error al parsear fecha: {$fecha}", ['exception' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * Parsear fecha-hora - devuelve datetime completo en formato MySQL
     */
    protected function parsearFechaHora(?string $fechaHora): ?string
    {
        if (!$fechaHora) return null;
        
        try {
            // Convertir fecha ISO 8601 a formato MySQL datetime
            $date = new \DateTime($fechaHora);
            return $date->format('Y-m-d H:i:s');
        } catch (\Exception $e) {
            Log::warning("Error al parsear fecha-hora: {$fechaHora}", ['exception' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * Extraer Asset ID de la URL para referencia
     */
    private function extraerAssetIdDeUrl(): ?string
    {
        if (preg_match('/assets\/([^\/]+)\/data/', $this->urlApi, $matches)) {
            return $matches[1];
        }
        return null;
    }

    /**
     * Obtener información del asset antes de importar
     */
    public function obtenerInfoAsset(): array
    {
        try {
            $assetUrl = str_replace('/data/?format=json', '', $this->urlApi);
            
            $httpClient = Http::timeout(30);

            // Usar las mismas credenciales configuradas
            if ($this->username && $this->password) {
                $httpClient = $httpClient->withBasicAuth($this->username, $this->password);
            }
            
            if (!empty($this->headers)) {
                $httpClient = $httpClient->withHeaders($this->headers);
            }
            
            $response = $httpClient->get($assetUrl);
            
            if (!$response->successful()) {
                return [
                    'success' => false, 
                    'message' => $this->getErrorMessage($response->status())
                ];
            }
            
            $data = $response->json();

            return [
                'success' => true,
                'name' => $data['name'] ?? 'Sin nombre',
                'deployment_count' => $data['deployment__submission_count'] ?? 0,
                'date_modified' => $data['date_modified'] ?? null,
                'asset_type' => $data['asset_type'] ?? 'survey'
            ];
            
        } catch (\Exception $e) {
            Log::error("Error al obtener info del asset: {$e->getMessage()}");
            return [
                'success' => false, 
                'message' => 'Error al obtener información: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Validar credenciales antes de importar
     */
    public function validarCredenciales(): bool
    {
        $info = $this->obtenerInfoAsset();
        return $info['success'] ?? false;
    }
}