<?php
// app/Services/ImportadorBase.php

namespace App\Services;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;

abstract class ImportadorBase
{
    protected string $urlApi;
    protected string $modelClass;
    protected string $idExterno;
    protected array $mapeoCampos;
    protected int $year;

    //Propiedades para autenticar
    protected ?string $username;
    protected ?string $password;
    protected array $headers = [];

    public function __construct(int $year)
    {
        $this->year = $year;
        $this->username = env('KOBO_USER_NAME');
        $this->password = env('KOBO_PASSWORD');
    }

    // Metodo principal que orquesta todo el proceso,
    public function importar()
    {
        try {
            $httpClient = Http::timeout(120);

            // Autenticación basica
            if($this->username && $this->password){
                $httpClient = $httpClient->withBasicAuth($this->username, $this->password);
            }

            if (!empty($this->headers)) {
                $httpClient = $httpClient->withHeaders($this->headers);
            }

            $response = $httpClient->get($this->urlApi);
            
            if (!$response->successful()) {
                $errorMessage = $this->getErrorMessage($response->status());
                Log::error("Error API: {$response->status()} - {$errorMessage} - URL: {$this->urlApi}");
                return [
                    'success' => false,
                    'message' => $errorMessage,
                    'status' => $response->status()
                ];
            }

            $datos = $response->json();
            $resultados = $this->extraerResultados($datos);
            
            if (empty($resultados)) {
                Log::info("No hay datos para importar de: {$this->urlApi}");
                return [
                    'success' => true,
                    'message' => 'No se encontraron datos para importar',
                    'total' => 0
                ];
            }

            $this->procesarDatos($resultados);
            return [
                'success' => true,
                'message' => 'Datos importados correctamente',
                'total' => count($resultados)
            ];

        } catch (Exception $e) {
            Log::error("Excepción en importar: {$e->getMessage()} - URL: {$this->urlApi}");
            return [
                'success' => false,
                'message' => 'Error interno: ' . $e->getMessage(),
                'exception' => get_class($e)
            ];
        }
    }
    /**
     * Extraer los resultados de la respuesta API
     * Las clases hijas pueden sobrescribir este método para manejar diferentes estructuras
     */
    protected function extraerResultados(array $datos): array
    {
        // Por defecto, busca en 'results', si no existe, devuelve todos los datos
        return $datos['results'] ?? $datos;
    }

    protected function procesarDatos(array $datosApi)
    {
        foreach ($datosApi as $dato) {
            $this->procesarRegistro($dato);
        }
    }

    protected function procesarRegistro(array $datoApi)
    {
        $datosMapeados = $this->mapearDatos($datoApi);
        $datosMapeados['version'] = $this->year;

        // Normaliza solo si es array/objeto/Collection
        foreach ($datosMapeados as $k => $v) {
            if (is_array($v) || $v instanceof \Illuminate\Support\Collection || is_object($v)) {
                $datosMapeados[$k] = $this->normalizarValor($v);
            }
        }

        // Obtener el valor del ID externo, manejando estructuras anidadas
        $valorIdExterno = $this->obtenerValorIdExterno($datoApi);

        $model = $this->modelClass::where($this->idExterno, $valorIdExterno)
                                  ->where('version', $this->year)
                                  ->first();

        if ($model) {
            $model->update($datosMapeados);
            Log::info("Actualizado registro con ID: {$valorIdExterno}");
        } else {
            $this->modelClass::create($datosMapeados);
            Log::info("Creado nuevo registro con ID: {$valorIdExterno}");
        }
    }

    /**
     * Obtener el valor del ID externo, manejando campos anidados
     */
    protected function obtenerValorIdExterno(array $datoApi)
    {
        // Si el idExterno contiene '/', es un campo anidado
        if (strpos($this->idExterno, '/') !== false) {
            $campos = explode('/', $this->idExterno);
            $valor = $datoApi;
            
            foreach ($campos as $campo) {
                $valor = $valor[$campo] ?? null;
                if ($valor === null) break;
            }
            
            return $valor;
        }
        
        return $datoApi[$this->idExterno] ?? null;
    }
    /**
     * Establecer credenciales para autenticación
     */
    public function setCredenciales(?string $username, ?string $password): self
    {
        $this->username = $username;
        $this->password = $password;
        return $this;
    }
     /**
     * Establecer headers personalizados
     */
    public function setHeaders(array $headers): self
    {
        $this->headers = array_merge($this->headers, $headers);
        return $this;
    }
    /**
     * Obtener mensaje de error basado en código HTTP
     */
    protected function getErrorMessage(int $statusCode): string
    {
        switch ($statusCode) {
            case 401:
                return 'Credenciales inválidas o token expirado';
            case 403:
                return 'No tienes permisos para acceder a este recurso';
            case 404:
                return 'Recurso no encontrado';
            case 429:
                return 'Demasiadas peticiones. Intenta más tarde';
            case 500:
                return 'Error interno del servidor';
            default:
                return "Error HTTP {$statusCode}";
        }
    }

    abstract protected function mapearDatos(array $datoApi): array;

    protected function normalizarValor($valor)
    {
        // Si es null, devolver null
        if ($valor === null) {
            return null;
        }
        
        if(is_array($valor)){
            return json_encode($valor, JSON_UNESCAPED_UNICODE);
        }

        if($valor instanceof \Illuminate\Support\Collection){
            return json_encode($valor->all(), JSON_UNESCAPED_UNICODE);
        }

        //Si es objeto, intentar convetir a string o json
        if(is_object($valor)){
            if(method_exists($valor, '_toString')){
                return (string) $valor;
            }
            return json_encode($valor, JSON_UNESCAPED_UNICODE);
        }
        // Si es booleano, convertir a string
        if (is_bool($valor)) {
            return $valor ? '1' : '0';
        }

        //ANTERIOR
        // if (is_array($valor)) {
        //     return json_encode($valor, JSON_UNESCAPED_UNICODE);
        // }

        // if ($valor instanceof \Illuminate\Support\Collection) {
        //     return json_encode($valor->all(), JSON_UNESCAPED_UNICODE);
        // }

        // if (is_object($valor)) {
        //     return method_exists($valor, '__toString')
        //         ? (string) $valor
        //         : json_encode($valor, JSON_UNESCAPED_UNICODE);
        // }

        return $valor; // escalares se guardan igual
    }
}