@extends('adminlte::page')

@section('title', 'Majomut - Vista de Inspección')

@section('content_header')
    <div class="d-flex justify-content-between align-items-center">
        <h1><i class="fas fa-eye"></i> Vista de Inspección por Productor</h1>
        <a href="/Inspeccio" class="btn btn-secondary">
            <i class="fas fa-arrow-left"></i> Volver a Lista
        </a>
    </div>
@stop

@section('content')
    <div class="container-fluid" id="inspeccion-container">
        
        <div class="card mb-3">
            <div class="card-body">
                <div class="row">
                    <div class="col-md-4">
                        <label>Seleccionar Productor</label>
                        <select class="form-control" id="select-productor">
                            <option value="">Cargando...</option>
                        </select>
                    </div>
                    <div class="col-md-4">
                        <label>Fecha de Inspección</label>
                        <select class="form-control" id="select-fecha">
                            <option value="">Seleccione un productor primero</option>
                        </select>
                    </div>
                    <div class="col-md-4">
                        <label>&nbsp;</label>
                        <div>
                            <button class="btn btn-primary btn-block" id="btn-cargar">
                                <i class="fas fa-search"></i> Ver Inspección
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        
        <div id="vista-inspeccion" style="display: none;">
            
            
            <div class="card border-primary mb-3">
                <div class="card-header bg-primary text-white">
                    <div class="row">
                        <div class="col-md-8">
                            <h3 class="mb-0"><i class="fas fa-user-circle"></i> <span id="header-nombre"></span></h3>
                            <small>DNI: <strong id="header-dni"></strong> | Inspector: <strong id="header-inspector"></strong> | Fecha: <strong id="header-fecha"></strong></small>
                        </div>
                        <div class="col-md-4 text-right">
                            <button class="btn btn-light btn-sm" onclick="window.print()">
                                <i class="fas fa-print"></i> Imprimir
                            </button>
                            <button class="btn btn-light btn-sm" id="btn-exportar-pdf">
                                <i class="fas fa-file-pdf"></i> PDF
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            
            <ul class="nav nav-tabs" id="inspeccionTabs" role="tablist">
                <li class="nav-item">
                    <a class="nav-link active" id="general-tab" data-toggle="tab" href="#general" role="tab">
                        <i class="fas fa-info-circle"></i> Información General
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" id="areas-tab" data-toggle="tab" href="#areas" role="tab">
                        <i class="fas fa-map"></i> Áreas y Manejo
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" id="produccion-tab" data-toggle="tab" href="#produccion" role="tab">
                        <i class="fas fa-industry"></i> Beneficio y Almacenamiento
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" id="capacitaciones-tab" data-toggle="tab" href="#capacitaciones" role="tab">
                        <i class="fas fa-graduation-cap"></i> Capacitaciones
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" id="parcelas-tab" data-toggle="tab" href="#parcelas" role="tab">
                        <i class="fas fa-th"></i> Parcelas (<span id="count-parcelas">0</span>)
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" id="resumen-tab" data-toggle="tab" href="#resumen" role="tab">
                        <i class="fas fa-chart-pie"></i> Resumen
                    </a>
                </li>
            </ul>

            <div class="tab-content" id="inspeccionTabContent">
                
                
                <div class="tab-pane fade show active" id="general" role="tabpanel">
                    <div class="row mt-3">
                        
                        
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header bg-info text-white">
                                    <h5 class="mb-0"><i class="fas fa-user"></i> Datos del Productor</h5>
                                </div>
                                <div class="card-body p-0">
                                    <table class="table table-sm table-striped mb-0 ficha-table">
                                        <tbody id="datos-productor"></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header bg-success text-white">
                                    <h5 class="mb-0"><i class="fas fa-map-marker-alt"></i> Ubicación y Estatus</h5>
                                </div>
                                <div class="card-body p-0">
                                    <table class="table table-sm table-striped mb-0 ficha-table">
                                        <tbody id="ubicacion-estatus"></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        
                        <div class="col-md-12 mt-3">
                            <div class="card">
                                <div class="card-header bg-warning">
                                    <h5 class="mb-0"><i class="fas fa-clipboard-check"></i> Información de la Inspección</h5>
                                </div>
                                <div class="card-body p-0">
                                    <table class="table table-sm table-striped mb-0 ficha-table">
                                        <tbody id="info-inspeccion"></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        
                        <div class="col-md-12 mt-3">
                            <div class="card">
                                <div class="card-header bg-secondary text-white">
                                    <h5 class="mb-0"><i class="fas fa-sitemap"></i> Participación en Otras Organizaciones</h5>
                                </div>
                                <div class="card-body p-0">
                                    <table class="table table-sm table-striped mb-0 ficha-table">
                                        <tbody id="otras-organizaciones"></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                    </div>
                </div>

                
                <div class="tab-pane fade" id="areas" role="tabpanel">
                    <div class="row mt-3">
                        
                        
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header bg-primary text-white">
                                    <h5 class="mb-0"><i class="fas fa-map"></i> Áreas Diferentes a Café</h5>
                                </div>
                                <div class="card-body p-0">
                                    <table class="table table-sm table-striped mb-0 ficha-table">
                                        <tbody id="areas-diferentes"></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        
                        <div class="col-md-6">
                            <div class="card">
                                <div class="card-header bg-success text-white">
                                    <h5 class="mb-0"><i class="fas fa-seedling"></i> Semillero y Vivero</h5>
                                </div>
                                <div class="card-body p-0">
                                    <table class="table table-sm table-striped mb-0 ficha-table">
                                        <tbody id="semillero-vivero"></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                       
                        <div class="col-md-12 mt-3">
                            <div class="card">
                                <div class="card-header text-white" style="background-color: #8B4513;">
                                    <h5 class="mb-0"><i class="fas fa-leaf"></i> Abonera y Fertilización</h5>
                                </div>
                                <div class="card-body p-0">
                                    <table class="table table-sm table-striped mb-0 ficha-table">
                                        <tbody id="abonera"></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        
                        <div class="col-md-12 mt-3">
                            <div class="card">
                                <div class="card-header bg-danger text-white">
                                    <h5 class="mb-0"><i class="fas fa-bug"></i> Control de Plagas y Conservación</h5>
                                </div>
                                <div class="card-body p-0">
                                    <table class="table table-sm table-striped mb-0 ficha-table">
                                        <tbody id="control-plagas"></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                    </div>
                </div>

                
                <div class="tab-pane fade" id="produccion" role="tabpanel">
                    <div class="row mt-3">
                        
                        
                        <div class="col-md-12">
                            <div class="card">
                                <div class="card-header bg-primary text-white">
                                    <h5 class="mb-0"><i class="fas fa-tint"></i> Beneficio Húmedo</h5>
                                </div>
                                <div class="card-body p-0">
                                    <table class="table table-sm table-striped mb-0 ficha-table">
                                        <tbody id="beneficio-humedo"></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        
                        <div class="col-md-12 mt-3">
                            <div class="card">
                                <div class="card-header bg-warning">
                                    <h5 class="mb-0"><i class="fas fa-warehouse"></i> Almacenamiento de Insumos</h5>
                                </div>
                                <div class="card-body p-0">
                                    <table class="table table-sm table-striped mb-0 ficha-table">
                                        <tbody id="almacenamiento"></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        
                        <div class="col-md-12 mt-3">
                            <div class="card">
                                <div class="card-header bg-success text-white">
                                    <h5 class="mb-0"><i class="fas fa-truck"></i> Cosecha y Comercialización</h5>
                                </div>
                                <div class="card-body p-0">
                                    <table class="table table-sm table-striped mb-0 ficha-table">
                                        <tbody id="cosecha-comercializacion"></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                    </div>
                </div>

                
                <div class="tab-pane fade" id="capacitaciones" role="tabpanel">
                    <div class="row mt-3">
                        <div class="col-md-12">
                            <div class="card">
                                <div class="card-header bg-info text-white">
                                    <h5 class="mb-0"><i class="fas fa-graduation-cap"></i> Capacitaciones y Conocimientos</h5>
                                </div>
                                <div class="card-body p-0">
                                    <table class="table table-sm table-striped mb-0 ficha-table">
                                        <tbody id="capacitaciones-table"></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

               
                <div class="tab-pane fade" id="parcelas" role="tabpanel">
                    <div class="mt-3">
                        <div id="parcelas-container"></div>
                    </div>
                </div>

                
                <div class="tab-pane fade" id="resumen" role="tabpanel">
                    <div class="row mt-3">
                        
                        
                        <div class="col-md-12 mb-3">
                            <div class="row text-center">
                                <div class="col-md-3">
                                    <div class="card bg-primary text-white">
                                        <div class="card-body">
                                            <h3 id="resumen-parcelas">0</h3>
                                            <p class="mb-0">Parcelas Totales</p>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="card bg-success text-white">
                                        <div class="card-body">
                                            <h3 id="resumen-area">0</h3>
                                            <p class="mb-0">Hectáreas Totales</p>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="card bg-info text-white">
                                        <div class="card-body">
                                            <h3 id="resumen-plantas">0</h3>
                                            <p class="mb-0">Plantas Productivas</p>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <div class="card bg-warning text-white">
                                        <div class="card-body">
                                            <h3 id="resumen-kg">0</h3>
                                            <p class="mb-0">Kg Estimados</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        
                        <div class="col-md-12">
                            <div class="card">
                                <div class="card-header bg-success text-white">
                                    <h5 class="mb-0"><i class="fas fa-chart-line"></i> Resumen de Producción</h5>
                                </div>
                                <div class="card-body p-0">
                                    <table class="table table-sm table-striped mb-0 ficha-table">
                                        <tbody id="resumen-produccion"></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        
                        <div class="col-md-12 mt-3">
                            <div class="card">
                                <div class="card-header bg-warning">
                                    <h5 class="mb-0"><i class="fas fa-comment-alt"></i> Observaciones Generales</h5>
                                </div>
                                <div class="card-body">
                                    <div id="resumen-observaciones" class="alert alert-info mb-0">
                                        Sin observaciones
                                    </div>
                                </div>
                            </div>
                        </div>

                        
                        <div class="col-md-12 mt-3">
                            <div class="card">
                                <div class="card-header bg-primary text-white">
                                    <h5 class="mb-0"><i class="fas fa-chart-bar"></i> Gráfica de Producción por Parcela</h5>
                                </div>
                                <div class="card-body">
                                    <canvas id="grafica-parcelas" height="80"></canvas>
                                </div>
                            </div>
                        </div>

                    </div>
                </div>

            </div>
        </div>

    </div>
@stop

@section('css')
    <style>
        
        .ficha-table tbody tr td:first-child {
            font-weight: 600;
            background-color: #f8f9fa;
            width: 40%;
            padding: 8px 12px;
            border-right: 2px solid #dee2e6;
        }
        
        .ficha-table tbody tr td:last-child {
            padding: 8px 12px;
        }
        
        .ficha-table tbody tr:hover {
            background-color: #f1f3f5;
        }
        
        
        .badge-lg {
            padding: 8px 12px;
            font-size: 14px;
        }
        
        
        .parcela-card {
            border-left: 4px solid #007bff;
            margin-bottom: 20px;
            transition: all 0.3s;
        }
        
        .parcela-card:hover {
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        
        .parcela-card.activa {
            border-left-color: #28a745;
        }
        
        .parcela-card.inactiva {
            border-left-color: #dc3545;
            opacity: 0.85;
        }
        
        
        .icon-si {
            color: #28a745;
            font-weight: bold;
        }
        
        .icon-no {
            color: #dc3545;
            font-weight: bold;
        }
        
        
        @media print {
            .no-print, .nav-tabs, button {
                display: none !important;
            }
            .card {
                page-break-inside: avoid;
                border: 1px solid #000 !important;
            }
            .tab-content > .tab-pane {
                display: block !important;
                opacity: 1 !important;
            }
        }
        
        
        .valor-vacio {
            color: #6c757d;
            font-style: italic;
        }
        
        
        .section-divider {
            background-color: #e9ecef;
            font-weight: bold;
            text-transform: uppercase;
            font-size: 11px;
            color: #495057;
        }
    </style>
@stop

@section('js')
    <script src="https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.js"></script>
    <script>
        let inspeccionActual = null;
        let chartParcelas = null;

        $(document).ready(function() {
            cargarProductores();
            
            $('#select-productor').on('change', function() {
                const dni = $(this).val();
                if (dni) {
                    cargarFechasInspeccion(dni);
                }
            });

            $('#btn-cargar').on('click', function() {
                const fecha = $('#select-fecha').val();
                if (fecha) {
                    cargarInspeccion(fecha);
                }
            });
        });

        function cargarProductores() {
            $.get('/Inspeccio/productores', function(data) {
                const $select = $('#select-productor');
                $select.empty().append('<option value="">Seleccione un productor...</option>');
                
                data.forEach(function(prod) {
                    $select.append(`<option value="${prod.dni}">${prod.nombre} (${prod.dni})</option>`);
                });
            }).fail(function() {
                showNotification('Error al cargar productores', 'error');
            });
        }

        function cargarFechasInspeccion(dni) {
            $.get(`/Inspeccio/fechas/${dni}`, function(data) {
                const $select = $('#select-fecha');
                $select.empty().append('<option value="">Seleccione una fecha...</option>');
                
                data.forEach(function(item) {
                    $select.append(`<option value="${item.id}">${item.fecha} - ${item.inspector}</option>`);
                });
            }).fail(function() {
                showNotification('Error al cargar fechas', 'error');
            });
        }

        function cargarInspeccion(id) {
            $.get(`/Inspeccio/ver/${id}`, function(data) {
                inspeccionActual = data;
                renderizarInspeccion(data);
                $('#vista-inspeccion').fadeIn();
                $('html, body').animate({ scrollTop: $('#vista-inspeccion').offset().top - 100 }, 500);
            }).fail(function() {
                showNotification('Error al cargar inspección', 'error');
            });
        }

        function renderizarInspeccion(data) {
            
            $('#header-nombre').text(data.socio || 'Sin nombre');
            $('#header-dni').text(data.dni || '-');
            $('#header-inspector').text(data.inspector || '-');
            $('#header-fecha').text(formatDate(data.fecha_visita));

            
            renderDatosProductor(data);
            renderUbicacionEstatus(data);
            renderInfoInspeccion(data);
            renderOtrasOrganizaciones(data);
            renderAreasDiferentes(data);
            renderSemilleroVivero(data);
            renderAbonera(data);
            renderControlPlagas(data);
            renderBeneficioHumedo(data);
            renderAlmacenamiento(data);
            renderCosechaComercializacion(data);
            renderCapacitaciones(data);
            renderParcelas(data.parcelas || []);
            renderResumen(data);
        }

        
        function crearFila(pregunta, respuesta, destacar = false) {
            const valorDisplay = respuesta === null || respuesta === undefined || respuesta === '' 
                ? '<span class="valor-vacio">Sin información</span>' 
                : respuesta;
            
            const claseDestacada = destacar ? 'table-primary' : '';
            return `<tr class="${claseDestacada}">
                <td>${pregunta}</td>
                <td>${valorDisplay}</td>
            </tr>`;
        }

        function crearSeparador(titulo) {
            return `<tr class="section-divider">
                <td colspan="2">${titulo}</td>
            </tr>`;
        }

        function siNo(valor) {
            if (valor === 'si' || valor === 'Si' || valor === 'SI' || valor === true || valor === '1') {
                return '<span class="icon-si">✓ Sí</span>';
            } else if (valor === 'no' || valor === 'No' || valor === 'NO' || valor === false || valor === '0') {
                return '<span class="icon-no">✗ No</span>';
            }
            return '<span class="valor-vacio">Sin información</span>';
        }

        function formatArray(arr) {
            if (!arr) return '<span class="valor-vacio">Sin información</span>';
            if (Array.isArray(arr)) {
                return arr.length > 0 ? arr.join(', ') : '<span class="valor-vacio">Sin información</span>';
            }
            return arr;
        }

        function renderDatosProductor(data) {
            let html = '';
            html += crearFila('Nombre completo', data.socio);
            html += crearFila('DNI / Identificación', data.dni);
            html += crearFila('CURP', data.curp);
            html += crearFila('RFC', data.rfc);
            html += crearFila('INE', data.ine);
            html += crearFila('Folio Cafetalero', data.folio_cafetalero);
            html += crearFila('Folio SURI', data.folio_suri);
            html += crearFila('Género', data.genero);
            html += crearFila('Fecha de Nacimiento', formatDate(data.fecha_nacimiento));
            html += crearFila('Fecha de Ingreso', formatDate(data.fecha_ingreso));
            html += crearFila('Teléfono', data.telefono);
            
            $('#datos-productor').html(html);
        }

        function renderUbicacionEstatus(data) {
            const estatusBadge = data.estatus === 'Organico' 
                ? '<span class="badge badge-success badge-lg">Orgánico</span>'
                : data.estatus === 'Transicion'
                ? '<span class="badge badge-warning badge-lg">En Transición</span>'
                : data.estatus
                ? `<span class="badge badge-secondary badge-lg">${data.estatus}</span>`
                : '<span class="valor-vacio">Sin información</span>';
            
            let html = '';
            html += crearFila('Municipio', data.municipio);
            html += crearFila('Localidad', data.localidad);
            html += crearFila('Grupo de Trabajo', data.grupo_trabajo);
            html += crearFila('Coordenadas GPS', data.gps_productor);
            html += crearFila('Estatus', estatusBadge, true);
            html += crearFila('Fecha de Transición', formatDate(data.fecha_transicion));
            html += crearFila('Tenencia de Tierra', formatArray(data.tenencia_tierra));
            html += crearFila('¿Tiene título de tenencia?', siNo(data.titulo_tenencia));
            html += crearFila('Papel de tenencia', data.papel_tenencia);
            html += crearFila('¿Productor en baja?', siNo(data.baja_productor));
            
            $('#ubicacion-estatus').html(html);
        }

        function renderInfoInspeccion(data) {
            let html = '';
            html += crearFila('Fecha de Visita', formatDate(data.fecha_visita), true);
            html += crearFila('Hora de Inicio', data.hora_inicio);
            html += crearFila('Hora Final', data.hora_final);
            html += crearFila('Inspector', data.inspector);
            html += crearFila('ID de Inspección', data.id_inspeccion);
            html += crearFila('¿Productor nuevo?', siNo(data.productor_nuevo));
            html += crearFila('Fecha de cosecha orgánico', formatDate(data.fecha_cosecha_organico));
            
            $('#info-inspeccion').html(html);
        }

        function renderOtrasOrganizaciones(data) {
            let html = '';
            html += crearSeparador('Participación del Productor');
            html += crearFila('¿Participa en otra organización?', siNo(data.participa_otra_organizacion));
            html += crearFila('Nombre de otra organización', data.nombre_otra_organizacion);
            html += crearFila('Producto en otra organización', data.producto_otra_organizacion);
            html += crearFila('Otra organización (001)', data.nombre_otra_organizacion_001);
            
            html += crearSeparador('Participación de la Familia');
            html += crearFila('¿Familia participa en otra organización?', siNo(data.familia_participa_otra_organizacion));
            html += crearFila('Producto de la familia', data.familia_producto_otra_organizazacion);
            html += crearFila('Organización de la familia', data.familia_otra_organizacion);
            html += crearFila('Categoría de la familia', data.familia_categoria_otra_organizacion);
            
            $('#otras-organizaciones').html(html);
        }

        function renderAreasDiferentes(data) {
            let html = '';
            html += crearSeparador('Área de Autoconsumo');
            html += crearFila('¿Tiene área de autoconsumo?', siNo(data.tiene_area_autoconsumo));
            html += crearFila('Área de autoconsumo (ha)', data.area_autoconsumo);
            html += crearFila('Número de parcelas', data.autoconsumo_n_parcelas);
            html += crearFila('¿Usa químicos en autoconsumo?', siNo(data.autoconsumo_quimicos));
            html += crearFila('Fecha de uso de químicos', formatDate(data.autoconsumo_fecha_quimicos));
            
            html += crearSeparador('Área de Potrero');
            html += crearFila('¿Tiene área de potrero?', siNo(data.tiene_area_potrero));
            html += crearFila('Área de potrero (ha)', data.area_potrero);
            html += crearFila('Número de parcelas', data.potrero_n_parcelas);
            
            html += crearSeparador('Área de Bosque');
            html += crearFila('¿Tiene área de bosque?', siNo(data.tiene_area_bosque));
            html += crearFila('Área de bosque (ha)', data.area_bosque);
            html += crearFila('Número de parcelas', data.bosque_n_parcelas);
            
            html += crearSeparador('Área de Acahual');
            html += crearFila('¿Tiene área de acahual?', siNo(data.tiene_area_acahual));
            html += crearFila('Área de acahual (ha)', data.area_acahual);
            html += crearFila('Número de parcelas', data.acahual_n_parcelas);
            
            $('#areas-diferentes').html(html);
        }

        function renderSemilleroVivero(data) {
            let html = '';
            html += crearFila('¿Cuenta con semillero?', siNo(data.sem_semillero));
            html += crearFila('Número de semilleros/plantas', data.sem_numero_semilleros);
            html += crearFila('¿Semillas orgánicas?', siNo(data.sem_semillas_organicas));
            html += crearFila('Proveniencia de semillas', data.sem_proveniencia_semillas);
            html += crearFila('Variedades de semillas', formatArray(data.variedades_semillas));
            html += crearFila('¿Realiza control de plagas?', siNo(data.sem_control_plagas));
            html += crearFila('¿Aplica producto químico?', siNo(data.sem_aplica_producto_quimico));
            html += crearFila('Producto químico usado', data.sem_producto_quimico);
            html += crearFila('Fecha de aplicación química', formatDate(data.sem_fecha_aplicacion_quimico));
            html += crearFila('¿Aplica preparado orgánico?', siNo(data.sem_aplica_preparado));
            html += crearFila('Preparado usado', data.sem_preparado);
            html += crearFila('Fecha de aplicación preparado', formatDate(data.sem_fecha_aplicacion_preparado));
            
            $('#semillero-vivero').html(html);
        }

        function renderAbonera(data) {
            let html = '';
            html += crearSeparador('Producción de Abono');
            html += crearFila('¿Elaboró abono orgánico?', siNo(data.p1_abono_organico));
            html += crearFila('Tipo de abono orgánico', data.p1_tipo_abono_organico);
            html += crearFila('Plantas abonadas', formatArray(data.p1_plantas_abono_organico));
            html += crearFila('Kilos/litros de composta', data.p1_kilos_litros_abono_composta);
            html += crearFila('Kilos por planta', data.p1_kilos_planta_composta);
            html += crearFila('Litros de lixiviado', data.p1_cuantos_litros_lixiviado_lombricomposta);
            html += crearFila('Plantas con foliar', data.p1_plantas_foliar_lombricomposta);
            html += crearFila('Dónde aplicó composta', data.p1_donde_aplico_composta);
            html += crearFila('Kg cosechados de abono', data.p1_cosecha_abono);
            html += crearFila('Total plantas abonadas', data.p1_plantas_abonadas);
            
            html += crearSeparador('Compra de Abono');
            html += crearFila('¿Compra abono orgánico?', siNo(data.p1_compra_abono_organico));
            html += crearFila('¿Cumple con normativas?', siNo(data.p1_compra_abono_cumplimiento));
            
            html += crearSeparador('Uso de Estiércol');
            html += crearFila('¿Aplica estiércol crudo?', siNo(data.p1_aplica_estiercol_crudo));
            html += crearFila('Plantas con estiércol', data.p1_estiercol_crudo_plantas);
            html += crearFila('Kilos de estiércol', data.p1_estiercol_crudo_kg);
            
            html += crearSeparador('Vivero');
            html += crearFila('¿Usa composta en vivero?', siNo(data.p1_composta_vivero));
            html += crearFila('Bolsas en vivero', data.p1_composta_vivero_bolsas);
            
            $('#abonera').html(html);
        }

        function renderControlPlagas(data) {
            let html = '';
            html += crearSeparador('Control de Plagas General');
            html += crearFila('¿Realiza control de plagas?', siNo(data.control_plagas));
            html += crearFila('Insumos usados', formatArray(data.insumos_plagas_seleccionados));
            html += crearFila('Fecha de aplicación', formatDate(data.control_plagas_fecha));
            html += crearFila('¿Insumos aprobados?', siNo(data.insumos_aprobados));
            html += crearFila('¿Cuenta con aspersora?', siNo(data.aspersora));
            
            html += crearSeparador('Conservación de Agua y Suelo');
            html += crearFila('¿Realiza conservación de agua?', siNo(data.conservacion_agua));
            html += crearFila('Actividades de conservación', formatArray(data.conservacion_agua_actividades));
            
            html += crearSeparador('Uso de Ceniza');
            html += crearFila('¿Aplica ceniza?', siNo(data.aplica_ceniza));
            html += crearFila('Plantas con ceniza', data.ceniza_plantas);
            html += crearFila('Kilos de ceniza', data.ceniza_kg);
            html += crearFila('Origen de la ceniza', data.ceniza_origen);
            html += crearFila('¿Ceniza tratada?', siNo(data.ceniza_tratada));
            
            html += crearSeparador('Manejo de Residuos');
            html += crearFila('¿Hay basura inorgánica?', siNo(data.basura_inorganica));
            
            $('#control-plagas').html(html);
        }

        function renderBeneficioHumedo(data) {
            let html = '';
            html += crearSeparador('Proceso de Beneficio');
            html += crearFila('Método de fermentación', data.bh_fermentacion);
            html += crearFila('Fuente de agua para lavado', data.bh_fuente_agua_lavado);
            html += crearFila('¿Agua potable para lavado?', siNo(data.bh_lavado_agua_potable));
            html += crearFila('Tipo de despulpadora', data.bh_despulpadora_tipo);
            html += crearFila('¿Despulpadora exclusiva?', siNo(data.bh_despulpadora_exclusiva));
            html += crearFila('Método de secado', data.bh_secado);
            
            html += crearSeparador('Manejo de Residuos');
            html += crearFila('Destino del agua miel', data.bh_destino_agua_miel);
            html += crearFila('Destino de la pulpa', data.bh_destino_pulpa);
            
            html += crearSeparador('Riesgos y Prevención');
            html += crearFila('¿Riesgo de contaminación?', siNo(data.bh_riesgo_contaminacion));
            html += crearFila('Área de riesgo', data.bh_riesgo_contaminacion_area);
            html += crearFila('¿Riesgo de mezcla de producto?', siNo(data.bh_riesgos_mezcla_producto));
            html += crearFila('¿Instalaciones compartidas?', siNo(data.bh_instalaciones_compartidas));
            html += crearFila('Medidas de prevención', data.bh_medidas_prevencion);
            
            $('#beneficio-humedo').html(html);
        }

        function renderAlmacenamiento(data) {
            let html = '';
            html += crearSeparador('Infraestructura');
            html += crearFila('¿Tiene bodega especial?', siNo(data.bodega_especial));
            html += crearFila('¿Almacenamiento exclusivo?', siNo(data.almacenamiento_exclusivo));
            html += crearFila('¿Envases exclusivos?', siNo(data.envases_exclusivos));
            
            html += crearSeparador('Prevención de Contaminación');
            html += crearFila('Medidas de prevención', formatArray(data.prevencion_contaminacion_almacenamiento));
            html += crearFila('¿Riesgo de mezcla?', siNo(data.riesgos_mezcla_producto));
            html += crearFila('¿Riesgo con costales?', siNo(data.riegos_costales));
            
            html += crearSeparador('Control de Plagas');
            html += crearFila('¿Prevención de plagas?', siNo(data.prevencion_plagas_almacen));
            html += crearFila('Descripción de prevención', data.prevencion_plagas_almacen_describir);
            
            $('#almacenamiento').html(html);
        }

        function renderCosechaComercializacion(data) {
            let html = '';
            html += crearFila('Fecha de cosecha orgánica', formatDate(data.fecha_cosecha_organico));
            html += crearFila('Entrega de café a', formatArray(data.entrega_cafe));
            html += crearFila('Tipo de procesamiento', data.procesamiento);
            html += crearFila('¿Vende a otro productor?', siNo(data.venta_otro_productor));
            html += crearFila('Nombre del otro productor', data.otro_productor);
            html += crearFila('Kg cosecha pasada', data.kg_cosecha_pasada);
            
            $('#cosecha-comercializacion').html(html);
        }

        function renderCapacitaciones(data) {
            let html = '';
            html += crearSeparador('Capacitaciones Recibidas');
            html += crearFila('¿Recibió capacitaciones?', siNo(data.cap_capacitaciones));
            html += crearFila('¿Tiene programa individual?', siNo(data.cap_programa_individual));
            html += crearFila('¿Registro del promotor?', siNo(data.cap_registro_promotor));
            html += crearFila('¿Capacitación recibida?', siNo(data.cap_capacitacion_recibida));
            html += crearFila('Fecha de capacitación', formatDate(data.cap_capacitacion_fecha));
            html += crearFila('Temas de capacitación', formatArray(data.cap_capacitacion_temas));
            
            html += crearSeparador('Conocimientos');
            html += crearFila('¿Practica plan de manejo orgánico?', siNo(data.cap_practica_plan_manejo_organico));
            html += crearFila('¿Conoce insumos permitidos?', siNo(data.cap_conoce_insumos_permitidos));
            html += crearFila('¿Conoce reglamento interno?', siNo(data.cap_conoce_reglamento_interno));
            html += crearFila('¿Guarda recibos de ventas?', siNo(data.cap_recibos_ventas));
            
            html += crearSeparador('Premio Social');
            html += crearFila('¿Conoce el premio social?', siNo(data.conocimiento_premio_social));
            html += crearFila('¿Recibe premio social?', siNo(data.recibe_premio_social));
            html += crearFila('Uso del premio social', data.uso_premio_social);
            
            $('#capacitaciones-table').html(html);
        }

        function renderParcelas(parcelas) {
            $('#count-parcelas').text(parcelas.length);
            
            if (parcelas.length === 0) {
                $('#parcelas-container').html('<div class="alert alert-info">No hay parcelas registradas</div>');
                return;
            }
            
            let html = '';
            parcelas.forEach((parcela, index) => {
                const activa = parcela.activa === '0' || parcela.activa === 0;
                const estadoClass = activa ? 'activa' : 'inactiva';
                const estadoText = activa ? 'ACTIVA' : 'INACTIVA';
                const estadoBadge = activa ? 'success' : 'danger';
                
                html += `
                    <div class="card parcela-card ${estadoClass}">
                        <div class="card-header">
                            <div class="row">
                                <div class="col-md-8">
                                    <h5 class="mb-0">
                                        <i class="fas fa-map-marked-alt"></i> 
                                        Parcela ${parcela.numero}: ${parcela.nombre || 'Sin nombre'}
                                    </h5>
                                    <small class="text-muted">Código: ${parcela.codigo || '-'}</small>
                                </div>
                                <div class="col-md-4 text-right">
                                    <span class="badge badge-${estadoBadge} badge-lg">${estadoText}</span>
                                </div>
                            </div>
                        </div>
                        <div class="card-body p-0">
                            <table class="table table-sm table-striped mb-0 ficha-table">
                                <tbody>
                                    ${crearSeparador('Información General')}
                                    ${crearFila('Superficie (ha)', parcela.superficie)}
                                    ${crearFila('¿Variación en la parcela?', siNo(parcela.variacion_parcela))}
                                    ${crearFila('Área actual (ha)', parcela.area_actual)}
                                    ${crearFila('Explicación de variación', parcela.variacion_explicacion)}
                                    
                                    ${crearSeparador('Características')}
                                    ${crearFila('Variedades de café', formatArray(parcela.variedades))}
                                    ${crearFila('Tipo de sombra', formatArray(parcela.tipo_sombra))}
                                    ${crearFila('Estado de la sombra', parcela.estado_sombra)}
                                    ${crearFila('¿Tiene árboles altos?', siNo(parcela.arboles_altos))}
                                    ${crearFila('Número de árboles altos', parcela.arboles_altos_numero)}
                                    ${crearFila('¿Talas innecesarias?', siNo(parcela.talas_inecesaria))}
                                    ${crearFila('¿Renovación de sombras?', siNo(parcela.renovacion_sombras))}
                                    ${crearFila('¿Colindante con talas?', siNo(parcela.colindante_talas))}
                                    ${crearFila('Materia orgánica', parcela.materia_organica)}
                                    ${crearFila('Cobertura del suelo', parcela.cobertura)}
                                    ${crearFila('¿Hay deslavado?', siNo(parcela.deslavado))}
                                    
                                    ${crearSeparador('Prácticas de Conservación')}
                                    ${crearFila('¿Prácticas de conservación?', siNo(parcela.practicas_conservacion))}
                                    ${crearFila('¿Barreras vivas?', siNo(parcela.barreras_vivas))}
                                    ${crearFila('Metros de barreras', parcela.metros_barreras_vivas)}
                                    ${crearFila('¿Tiene terrazas?', siNo(parcela.terrazas))}
                                    ${crearFila('Número de terrazas', parcela.numero_terrazas)}
                                    ${crearFila('¿Mantenimiento?', siNo(parcela.mantemimiento))}
                                    ${crearFila('Descripción del mantenimiento', parcela.mantenimiento_descripcion)}
                                    
                                    ${crearSeparador('Manejo y Fertilización')}
                                    ${crearFila('¿Aplica estiércol crudo?', siNo(parcela.aplica_estiercol_crudo))}
                                    ${crearFila('Plantas con estiércol', parcela.estiercol_crudo_plantas)}
                                    ${crearFila('Kg de estiércol', parcela.estiercol_crudo_kg)}
                                    ${crearFila('¿Realiza podas?', siNo(parcela.podas))}
                                    ${crearFila('Descripción de podas', formatArray(parcela.podas_descripcion))}
                                    
                                    ${crearSeparador('Plagas y Enfermedades')}
                                    ${crearFila('Plagas/enfermedades presentes', formatArray(parcela.plagas_enfermedades))}
                                    ${crearFila('¿Aplica químico?', siNo(parcela.aplica_quimico))}
                                    ${crearFila('Producto químico usado', parcela.producto_quimico)}
                                    ${crearFila('Fecha de aplicación', formatDate(parcela.fecha_producto_quimico))}
                                    
                                    ${crearSeparador('Condiciones y Riesgos')}
                                    ${crearFila('Condiciones generales', parcela.condiciones)}
                                    ${crearFila('¿Hay basura?', siNo(parcela.basura))}
                                    ${crearFila('Nivel de riesgo', parcela.riesgo)}
                                    
                                    ${crearSeparador('Terrenos Colindantes')}
                                    ${crearFila('Norte', parcela.norte)}
                                    ${crearFila('Propietario norte', parcela.norte_propietario)}
                                    ${crearFila('Sur', parcela.sur)}
                                    ${crearFila('Propietario sur', parcela.sur_propietario)}
                                    ${crearFila('Este', parcela.este)}
                                    ${crearFila('Propietario este', parcela.este_propietario)}
                                    ${crearFila('Oeste', parcela.oeste)}
                                    ${crearFila('Propietario oeste', parcela.oeste_propietario)}
                                    
                                    ${crearSeparador('Riesgos de Contaminación')}
                                    ${crearFila('¿Herbicidas en colindantes?', siNo(parcela.herbicidas_colindantes))}
                                    ${crearFila('Nombre del herbicida', formatArray(parcela.nombre_herbicida_colindante))}
                                    ${crearFila('¿Riesgo de contaminación?', siNo(parcela.riesgos_contaminacion))}
                                    ${crearFila('Lado del riesgo', parcela.lado_riesgos_contaminacion)}
                                    ${crearFila('¿Franja de amortiguamiento?', siNo(parcela.franja_amortiguamiento))}
                                    ${crearFila('¿Tiene zanja?', siNo(parcela.zanja))}
                                    ${crearFila('¿Barreras vivas de protección?', siNo(parcela.barreras_vivas_proteccion))}
                                    ${crearFila('¿Sin medidas de protección?', siNo(parcela.ninguna_evita_contaminacion))}
                                    ${crearFila('¿Uso de glifosato?', siNo(parcela.glifosato))}
                                    ${crearFila('¿Aplicación de glifosato?', siNo(parcela.aplicacion_glifosato))}
                                    ${crearFila('Metros de aplicación glifosato', parcela.aplicacion_glifosato_metros)}
                                    
                                    ${crearSeparador('Estimación de Cosecha')}
                                    ${crearFila('Aspecto nivel 1', parcela.aspecto_nivel1)}
                                    ${crearFila('Aspecto nivel 2', parcela.aspecto_nivel2)}
                                    ${crearFila('Aspecto nivel 3', parcela.aspecto_nivel3)}
                                    ${crearFila('Plantas nivel 1', parcela.n_plantas_n1)}
                                    ${crearFila('Plantas nivel 2', parcela.n_plantas_n2)}
                                    ${crearFila('Plantas nivel 3', parcela.n_plantas_n3)}
                                    ${crearFila('Total plantas productivas', parcela.n_plantas_tot, true)}
                                    ${crearFila('Edad de plantas (años)', parcela.edad_plantas)}
                                    ${crearFila('¿Tiene plantillas?', siNo(parcela.plantillas))}
                                    ${crearFila('Número de plantillas', parcela.plantillas_numero)}
                                    ${crearFila('Kg pergamino nivel 1', parcela.kg_pergamino_n1)}
                                    ${crearFila('Kg pergamino nivel 2', parcela.kg_pergamino_n2)}
                                    ${crearFila('Kg pergamino nivel 3', parcela.kg_pergamino_n3)}
                                    ${crearFila('Total Kg estimados', `<strong>${parcela.kg_pergamino_tot || 0} kg</strong>`, true)}
                                    ${crearFila('Condiciones productivas', parcela.condiciones_productivas)}
                                    
                                    ${parcela.observaciones ? crearSeparador('Observaciones') : ''}
                                    ${parcela.observaciones ? crearFila('Observaciones', parcela.observaciones) : ''}
                                </tbody>
                            </table>
                        </div>
                    </div>
                `;
            });
            
            $('#parcelas-container').html(html);
        }

        function renderResumen(data) {
            
            $('#resumen-parcelas').text(data.parcelas_totales || 0);
            $('#resumen-area').text((data.area_total || 0) + ' ha');
            $('#resumen-plantas').text(data.plantas_totales || 0);
            $('#resumen-kg').text((data.kg_estimados_totales || 0) + ' kg');
            
            
            let html = '';
            html += crearFila('Parcelas totales', data.parcelas_totales || 0, true);
            html += crearFila('Área total (ha)', data.area_total || 0, true);
            html += crearFila('Plantas productivas totales', data.plantas_totales || 0, true);
            html += crearFila('Plantillas totales', data.plantillas_totales || 0);
            html += crearFila('Kg estimados totales', `<strong>${data.kg_estimados_totales || 0} kg</strong>`, true);
            html += crearFila('Kg cosecha pasada', data.kg_cosecha_pasada || 0);
            html += crearFila('Número de parcelas registradas', data.n_parcelas || 0);
            html += crearFila('¿Parcela nueva?', siNo(data.parcela_nueva));
            
            $('#resumen-produccion').html(html);
            
            
            $('#resumen-observaciones').html(data.observaciones_generales || 'Sin observaciones generales');
            
            
            if (data.razon_baja) {
                $('#resumen-observaciones').append(`<hr><strong>Razón de baja:</strong> ${data.razon_baja}`);
            }

            
            if (data.parcelas && data.parcelas.length > 0) {
                renderGraficaParcelas(data.parcelas);
            }
        }

        function renderGraficaParcelas(parcelas) {
            const ctx = document.getElementById('grafica-parcelas');
            if (!ctx) return;
            
            if (chartParcelas) {
                chartParcelas.destroy();
            }
            
            const labels = parcelas.map((p) => `P${p.numero}: ${(p.nombre || '').substring(0, 15)}`);
            const dataKg = parcelas.map(p => parseFloat(p.kg_pergamino_tot) || 0);
            const dataHa = parcelas.map(p => parseFloat(p.superficie) || 0);
            const dataPlantas = parcelas.map(p => parseFloat(p.n_plantas_tot) || 0);
            
            chartParcelas = new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: labels,
                    datasets: [
                        {
                            label: 'Kg Estimados',
                            data: dataKg,
                            backgroundColor: 'rgba(54, 162, 235, 0.7)',
                            borderColor: 'rgba(54, 162, 235, 1)',
                            borderWidth: 2,
                            yAxisID: 'y'
                        },
                        {
                            label: 'Hectáreas',
                            data: dataHa,
                            backgroundColor: 'rgba(75, 192, 192, 0.7)',
                            borderColor: 'rgba(75, 192, 192, 1)',
                            borderWidth: 2,
                            type: 'line',
                            yAxisID: 'y1'
                        }
                    ]
                },
                options: {
                    responsive: true,
                    interaction: {
                        mode: 'index',
                        intersect: false,
                    },
                    plugins: {
                        legend: {
                            position: 'top',
                        },
                        title: {
                            display: true,
                            text: 'Producción Estimada por Parcela'
                        }
                    },
                    scales: {
                        y: {
                            type: 'linear',
                            display: true,
                            position: 'left',
                            title: {
                                display: true,
                                text: 'Kilogramos'
                            }
                        },
                        y1: {
                            type: 'linear',
                            display: true,
                            position: 'right',
                            title: {
                                display: true,
                                text: 'Hectáreas'
                            },
                            grid: {
                                drawOnChartArea: false,
                            },
                        },
                    }
                }
            });
        }

        function formatDate(date) {
            if (!date) return '<span class="valor-vacio">Sin información</span>';
            try {
                
                let d;
                if (typeof date === 'object' && date.$date) {
                    d = new Date(date.$date);
                } else if (typeof date === 'number') {
                    d = new Date(date);
                } else {
                    d = new Date(date);
                }
                
                if (isNaN(d.getTime())) return date;
                
                return d.toLocaleDateString('es-ES', { 
                    year: 'numeric', 
                    month: 'long', 
                    day: 'numeric' 
                });
            } catch (e) {
                return date;
            }
        }

        function showNotification(message, type) {
            const alertClass = type === 'error' ? 'alert-danger' : 'alert-success';
            const notification = $(`
                <div class="alert ${alertClass} alert-dismissible fade show" style="position: fixed; top: 20px; right: 20px; z-index: 9999;">
                    ${message}
                    <button type="button" class="close" data-dismiss="alert">
                        <span>&times;</span>
                    </button>
                </div>
            `);
            $('body').append(notification);
            setTimeout(() => notification.alert('close'), 5000);
        }
    </script>
@stop